/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "norman.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// The games Norman knows how to play
#include "normancrazyeightsai.h"
#include "normancribbageai.h"
#include "normanheartsai.h"
#include "normanspadesai.h"
#include "normaneuchreai.h"

Norman::Norman(): AIBase()
{}

Norman::~Norman()
{}

CardSequence Norman::whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else if (game.contains("euchre"))
        cards = handleEuchreCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("crazy eights"))
        cards = handleCrazyEightsCards(game, playSequence, rules, hand, specialInformation);
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Norman::decision(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else if (game.contains("euchre"))
        bid = handleEuchreDecisions(game, playSequence, rules, hand, specialInformation, score);
    else if (game.contains("crazy eights"))
        bid = handleCrazyEightsDecisions(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::SkillLevel Norman::skillLevel(const QString& game) const
{
    if (game == "spades")
        return Player::Novice;
    else if (game == "cribbage")
        return Player::Amateur;
    else if (game == "hearts")
        return Player::Professional;
    else if (game == "old maid")
        return Player::Master;
    else if (game == "euchre")
        return Player::Novice;
    else if (game == "crazy eights")
        return Player::Apprentice;
    else
        return Player::Newb;
}

QString Norman::name() const
{
    return "Norman";
}

CardSequence Norman::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "cribbageDealerCrib")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.myCrib();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.opponentsCrib();
    }
    throw KardsGTError("Norman", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Norman::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards();
    }
    else if (game == "spadesExchangeCards")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Norman", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Norman::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        NormanHeartsAI ai(playSequence, rules, hand);
        return ai.selectCards();
    }
    else if (game == "heartsPassPhase")
    {
        NormanHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Norman", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Norman::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Norman", "handleSpadesBid", "Failed to determine game type or phase.");
}

int Norman::handleEuchreDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "euchreMakeTurnUp")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.turnUp(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreSelectTrump")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.selectTrump(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchrePlayAlone")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.playAlone(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreDefendAlone")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.defendAlone();
    }
    else if (game == "euchreWantsTurnUp")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.wantTurnUp();
    }
    throw KardsGTError("Norman", "handleEuchreDecisions", "Failed to determine game type or phase.");
}

CardSequence Norman::handleEuchreCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "euchre")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        Card partnersCard;

        partnersCard.setCard(specialInformation.value("Partner Played"));
        return ai.selectCards(partnersCard);
    }
    else if (game == "euchreDiscard")
    {
        NormanEuchreAI ai(playSequence, rules, hand);
        return ai.discard();
    }
    throw KardsGTError("Norman", "handleEuchreCards", "Failed to determine game type or phase.");
}

CardSequence Norman::handleCrazyEightsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "crazy eights SelectCard")
    {
        NormanCrazyEightsAI ai(playSequence, rules, hand, specialInformation);
        return ai.selectCard();
    }
    throw KardsGTError("Norman", "handleCrazyEightsCards", "Failed to determine game type or phase.");
}

int Norman::handleCrazyEightsDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "crazy eights SelectSuit")
    {
        NormanCrazyEightsAI ai(playSequence, rules, hand, specialInformation);
        return ai.selectSuit(score);
    }
    throw KardsGTError("Norman", "handleCrazyEightsDecisions", "Failed to determine game type or phase.");
}
