/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef KARDSGTINTERFACE_H
#define KARDSGTINTERFACE_H

#include "ui_kardsgtinterface.h"
#include "player.h"
#include "userprofiledatabase.h"

#include <vector>
using std::vector;
using std::pair;
#include <QWidget>

class GameFactory;
class PlayerFactory;
class QAssistantClient;

/**
 * This is the primary interface for the KardsGT card game. From this widget you can
 * start a game, modify settings, and exit the programme.
 * @note On the first run, the user name will be required to be entered.
 *
 * @author John Schneiderman
 */
class KardsGTInterface: public QMainWindow, public Ui_KardsGTInterface
{
    Q_OBJECT

public:
    /**
     * This is the default widget constructor.
     * @param fileName is the name of the game file we want to load.
     * @param parent is the parent of the widget.
     */
    KardsGTInterface(QString fileName="", QWidget *parent=0);
    /**
     * This is the default widget destructor.
     */
    virtual ~KardsGTInterface();

protected slots:
    /**
     * This slot sets up the game set in m_gamePlayed, and activates the game.
     * @throw KardsGTError if we fail to create the game.
     */
    void setupGame();
    /**
     * This slot opens a dialogue for the user to change the the programme settings.
     */
    void settingsConfigureKardsGT();
    /**
     * This slot opens a dialogue for the user to change their name.
     */
    void changeName();
    /**
     * This slot closes the current open game.
     */
    void closeOpenGame();
    /**
     * This slot opens the about information for the programme.
     */
    void helpAbout();
    /**
     * This slot re-starts the current game running.
     */
    void newGame();
    /**
     * This slot saves the game running.
     */
    void saveGame();
    /**
     * This slot opens a dialogue requesting the user to enter a filename for the programme to load.
     */
    void loadGame();
    /**
     * This slot opens the QAssistantClient and displays the handbook for KardsGT.
     */
    void helpHandbook();
    /**
     * This slot displays the about information for the QT system.
     */
    void helpAboutQT();
    /**
     * This slot starts the Cribbage game.
     */
    void turnOnCribbage();
    /**
     * This slot starts the Euchre game.
     */
    void turnOnEuchre();
    /**
     * This slot starts the Hearts game.
     */
    void turnOnHearts();
    /**
     * This slot starts the Old Maid game.
     */
    void turnOnOldMaid();
    /**
     * This slot starts the Spades game.
     */
    void turnOnSpades();
    /**
     * This slot starts the War game.
     */
    void turnOnWar();
    /**
     * This slot starts the Crazy Eights game.
     */
    void turnOnCrazyEights();

private:
    /// @param m_userProfileDB is the user game profile database.
    UserProfileDatabase m_userProfileDB;
    /// @param m_fileFilter is the filter for saved games for KardsGT.
    static const QString m_fileFilter;
    /// @param m_pAssistant is the QT assistant client.
    QAssistantClient *m_pAssistant;
    /// @param m_pPlayer is the current human player.
    Player *m_pPlayer;
    /// @param m_pPlayerFactory is the factory to create computer players.
    PlayerFactory *m_pPlayerFactory;
    /// @param m_pGameFactory is the factory to create a game widget.
    GameFactory *m_pGameFactory;
    /// @param m_loading indicates if we are currently loading a game.
    bool m_loading;
    /// @param m_gamePlayed is the string sentinal for each game widget.
    QString m_gamePlayed;
    /// @param m_opponents is a list of the names of the computer players.
    vector<QString> m_opponents;

    /**
     * Creates the AI Opponents in the opponent list.
     * @throw KardsGTError if we cannot create an opponent.
     */
    void createOpponents();
    /**
     * This retrieves and decodes a player's game skill level.
     * @param game is the name of the game to retrieve.
     * @return the skill level for that player at that game.
     */
    Player::SkillLevel playersLevel(const QString &game) const;
    /**
     * This loads up a card game from a file name.
     * @param filename is the name of the file for a card game.
     */
    void loadGame(QString filename);
    /**
     * This actually starts the game widget stored in the game factory.
     */
    void startGame();
    /**
     * This sets the menu actions to be enable or disabled based on the action value.
     * @param actionOn if true will enable all menu actions, if false will disable all the menu actions.
     */
    void setActions(bool actionOn);
    /**
     * This sets the m_gamePlayed and checks the appropriate menu action.
     * @param game is the game sentinal to enable.
     */
    void selectGameAction(const QString &game);
};
#endif
