/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "spades.h"
#include "cardproperties.h"
#include "card.h"
#include "kardsgterror.h"

SpadesRules::SpadesRules(): RuleBase()
{
    m_spadesBroken = false;
}

SpadesRules::~SpadesRules()
{}

bool SpadesRules::isGameOver(const PlayerList& players) const
{
    for (int index = 0; index < players.size(); ++index)
        if (players[index].score() >= GAME_OVER_SCORE)
            return true;
    return false;
}

bool SpadesRules::isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &player) const
{
    CardProperties playerHandProperties(player.hand());

    // Are we trying to lead
    if (sequence.isEmpty())
        if (cardPlayed.suit() == Card::SPADES) // Can only lead with a spade if spades has been broken or you only have are spades
            if (m_spadesBroken || playerHandProperties.suits(Card::SPADES).size() == player.hand().size())
                return true;
            else
                return false;
        else
            return true;
    // Is the player playing a card of the lead suit
    if (cardPlayed.suit() == sequence.front().suit())
        return true;
    // Is the player playing a card not of the lead suit because he doesn't have the suit
    if (playerHandProperties.suits(sequence.front().suit()).isEmpty())
        return true;
    return false;
}

bool SpadesRules::isPhaseOver(const PlayerList &, const CardSequence &playSequence) const
{
    if (playSequence.size() == PHASE_OVER)
        return true;
    else
        return false;
}

bool SpadesRules::isRoundOver(const PlayerList& players) const
{
    for (int i=0; i < players.size(); ++i)
        if (! players[i].hand().isEmpty())
            return false;
    return true;
}

int SpadesRules::cardValue(const Card &) const
{
    throw KardsGTError("SpadesRules", "cardValue", "This is not implemented!");
    return -1;
}

int SpadesRules::maximumNumberOfPlayers() const
{
    return MAXIMUM_NUMBER_OF_PLAYERS;
}

int SpadesRules::minimumNumberOfPlayers() const
{
    return MINIMUM_NUMBER_OF_PLAYERS;
}

int SpadesRules::numberOfCardsToDeal(int) const
{
    return CARDS_TO_DEAL;
}

int SpadesRules::winningGameScore(int) const
{
    return GAME_OVER_SCORE;
}

bool SpadesRules::isSpadesBroken() const
{
    return m_spadesBroken;
}

void SpadesRules::setSpadesBroken(bool spadesBroken)
{
    m_spadesBroken = spadesBroken;
}

int SpadesRules::rankValue(const Card& card) const
{
    int value = RANK_ERR;

    switch (card.rank())
    {
    case Card::ACE:
        value=ACE;
        break;
    case Card::TWO:
        value=TWO;
        break;
    case Card::THREE:
        value=THREE;
        break;
    case Card::FOUR:
        value=FOUR;
        break;
    case Card::FIVE:
        value=FIVE;
        break;
    case Card::SIX:
        value=SIX;
        break;
    case Card::SEVEN:
        value=SEVEN;
        break;
    case Card::EIGHT:
        value=EIGHT;
        break;
    case Card::NINE:
        value=NINE;
        break;
    case Card::TEN:
        value=TEN;
        break;
    case Card::JACK:
        value=JACK;
        break;
    case Card::QUEEN:
        value=QUEEN;
        break;
    case Card::KING:
        value=KING;
        break;
    default:
        value=RANK_ERR;
    }
    return value;
}
