/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CARIBBAGEBOARD_H
#define CARIBBAGEBOARD_H

#include <QWidget>
#include <QColor>
#include <QPaintEvent>
#include <vector>
using std::vector;
using std::pair;

class QSize;
class QPainter;
class QPoint;
class QLCDNumber;
class CribbageBlock;
class CribbagePeg;

/**
 * This is a widget to draw a cribbage board.
 *
 * @author John Schneiderman
 */
class CribbageBoard: public QWidget
{
    Q_OBJECT

public:
    /**
     * Default constructor.
     * @param parent is the CribbageBoard's parent widget.
     */
    CribbageBoard(QWidget *parent=0);
    /**
     * Destructor.
     */
    ~CribbageBoard();
    /**
     * Gives the scores for a specific colour.
     * @param pegColour is the colour of the CribbagePeg we want the scores for.
     * @return the first score is the lead CribbagePeg, and the second is the non-lead CribbagePeg.
     */
    pair<int, int> scores(const QColor &pegColour) const;
    /**
     * Sets the score for a specific CribbagePeg.
     * @param pegColour is the colour of the pegs we want to set.
     * @param scores is the scores we want to set for our pegs. The first score is the lead CribbagePeg, and the second score is the non-lead CribbagePeg.
     */
    void setScores(const QColor &pegColour, pair<int, int> scores);
    /**
     * This returns the cribbage board's preferred size.
     * @return the preferred size of the cribbage board.
     */
    QSize sizeHint() const;
    /**
     * This sets the score for a CribbagePeg.
     * @param pegColour is the colour of the CribbagePeg we wish to CribbagePeg.
     * @param score is the new score for the CribbagePeg.
     */
    void pegScore(const QColor &pegColour, int score);
    /**
     * This pegs the number of stakes won.
     * @param pegColour is the colour of the CribbagePeg we wish to CribbagePeg for.
     * @param stakes is the number of stakes we have won thus far.
     * @note if stakes is greater than NUMBER_OF_STAKES then we just return.
     */
    void pegStakes(const QColor &pegColour, int stakes);
    /**
     * Places the round pegs back in their starting positions.
     */
    void reset();
    /**
     * This sets the name to associate with a particular CribbagePeg colour.
     * @param pegColour is the colour of the CribbagePeg to set.
     * @param name is the name of the player that is associated with that CribbagePeg colour.
     */
    void setName(const QColor &pegColour, const QString &name);
    /**
     * Clears everything in the board.
     */
    void clear();

protected:
    /**
     * Causes the cribbage board to be drawn.
     * @param event is the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);

private:
    /**
     * These are some of the random constants used to create the Cribbage board.
     * @param PREFERRED_WIDTH is the PREFERRED width.
     * @param PREFERRED_HEIGHT is the PREFERRED height.
     * @param NUMBER_OF_PEGS is how many pegs each colour should have.
     * @param NUMBER_OF_STAKES is how many stakes needed to win the game.
     * @param NUMBER_PEGS_PER_BLOCK is how many pegs to place in each block.
     * @param NUMBER_VERTICAL_BLOCKS is how many vertical blocks to create for the board.
     */
    enum PrivateConstants
    {
        PREFERRED_WIDTH=240,
        PREFERRED_HEIGHT=480,
        NUMBER_OF_PEGS=3,
        NUMBER_OF_STAKES=7,
        NUMBER_PEGS_PER_BLOCK=5,
        NUMBER_VERTICAL_BLOCKS=7
    };
    /**
     * These are the scores that mark the beginning of a new pegging zone.
     * @param ZERO_ZONE_ENDING is the number where the zero pegging zone ends.
     * @param FIRST_ZONE_ENDING is the number where the first pegging zone ends.
     * @param SECOND_ZONE_ENDING is the number where the second pegging zone ends.
     * @param THIRD_ZONE_ENDING is the number where the third pegging zone ends.
     * @param FOURTH_ZONE_ENDING is the number where the fourth pegging zone ends.
     * @param FIFTH_ZONE_ENDING is the number where the fifth pegging zone ends.
     * @param SIXTH_ZONE_ENDING is the number where the sixth pegging zone ends.
     */
    enum PegZoneEndings
    {
        ZERO_ZONE_ENDING=1,
        FIRST_ZONE_ENDING=36,
        SECOND_ZONE_ENDING=46,
        THIRD_ZONE_ENDING=81,
        FOURTH_ZONE_ENDING=86,
        FIFTH_ZONE_ENDING=121,
        SIXTH_ZONE_ENDING=122
    };
    /**
     * These are the starting coordinates for our pegging zones. All values are inclusive.
     * @param BLOCK_OFFSET is the distance to the edge of the CribbagePeg.
     * @param ZERO_ZONE_X is the x-coordinate pegging zone for the starting positions on the red zone at 0.
     * @param ZERO_ZONE_Y is the y-coordinate pegging zone for the starting positions on the red zone at 0.
     * @param FIRST_ZONE_X is the x-coordinate pegging zone for 1-35 starting on the red zone at 35.
     * @param FIRST_ZONE_Y is the y-coordinate pegging zone for 1-35 starting on the red zone at 35.
     * @param SECOND_ZONE_X is the x-coordinate pegging zone for 36-45 starting on the red zone at 36.
     * @param SECOND_ZONE_Y is the y-coordinate pegging zone for 36-45 starting on the red zone at 36.
     * @param THIRD_ZONE_X is the x-coordinate pegging zone for 46-80 starting on the blue zone at 80.
     * @param THIRD_ZONE_Y is the y-coordinate pegging zone for 46-80 starting on the blue zone at 80.
     * @param FOURTH_ZONE_X is the x-coordinate pegging zone for 81-85 starting on the red zone at 85.
     * @param FOURTH_ZONE_Y is the y-coordinate pegging zone for 81-85 starting on the red zone at 85.
     * @param FIFTH_ZONE_X is the x-coordinate pegging zone for 86-120 starting on the red zone at 120.
     * @param FIFTH_ZONE_Y is the y-coordinate pegging zone for 86-120 starting on the red zone at 120.
     * @param SIXTH_ZONE_X is the x-coordinate pegging zone for 121.
     * @param SIXTH_ZONE_Y is the y-coordinate pegging zone for 121.
     * @param STAKE_ZONE_X is the x-coordinate for the stake pegs starting on the red CribbagePeg.
     * @param STAKE_ZONE_Y is the y-coordinate for the stake pegs starting on the red CribbagePeg.
     */
    enum PeggingZones
    {
        BLOCK_OFFSET=3,
        ZERO_ZONE_X=3, ZERO_ZONE_Y=75,
        FIRST_ZONE_X=3, FIRST_ZONE_Y=0,
        SECOND_ZONE_X=13, SECOND_ZONE_Y=-20,
        THIRD_ZONE_X=37, THIRD_ZONE_Y=0,
        FOURTH_ZONE_X=23, FOURTH_ZONE_Y=80,
        FIFTH_ZONE_X=17 , FIFTH_ZONE_Y=0,
        SIXTH_ZONE_X=20, SIXTH_ZONE_Y=-3,
        STAKE_ZONE_X=25, STAKE_ZONE_Y=90
    };

    /// @param m_redPegs are all of the red pegs.
    vector<CribbagePeg *> m_redPegs;
    /// @param m_bluePegs are all of the blue pegs.
    vector<CribbagePeg *> m_bluePegs;
    /// @param m_legendPegs is the pegs with the names of each player.
    vector<pair<CribbagePeg *, QString> > m_legendPegs;
    /// @param m_peggingsZones is the beginning of each pegging zone with the coordinates being the exact location for the first CribbagePeg in the series.
    vector<QPoint *> m_peggingsZones;
    /// @param m_blocks is the pegging blocks for our board.
    vector<CribbageBlock *> m_blocks;

    /**
     * This will move a CribbagePeg on the cribbage board.
     * @param CribbagePeg is the CribbagePeg to move.
     * @param newScore is the new score to set the CribbagePeg to.
     */
    void movePeg(CribbagePeg &CribbagePeg, int newScore);
    /**
     * This will draw the cribbage board.
     * @param painter is the device onto which we will draw the cribbage board.
     */
    void draw(QPainter *painter);
};
#endif
