/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "crazyeightsinterface.h"
#include "ui_crazyeightsinterface.h"
#include "kard.h"
#include "suitrequest.h"
#include "kardmessagedisplay.h"
#include "crazyeightsplayer.h"
#include "kardsgterror.h"

#include <QMessageBox>
#include <QString>
#include <QPushButton>
#include <QtDebug>

CrazyEightsInterface::CrazyEightsInterface(UserProfileDatabase &profileDatabase, QWidget *parent): QWidget(parent), CrazyEights(this, profileDatabase)
{
    Ui_CrazyEightsInterface::setupUi(this);
    m_pUserProfile = &profileDatabase;
    discardPile->setTopCardUp(true);

    //Set the colours for our game
    setBackgroundRole(QPalette::Background);
    setAutoFillBackground (true);
    QPalette pal(this->palette());
    pal.setColor(QPalette::Background, Qt::darkGreen);
    this->setPalette(pal);
    player3Cards->setVertical(true);
    player4Cards->setVertical(true);
    player7Cards->setVertical(true);

    // Setup pointers to interface components
    m_kardPlayers.push_back(player1);
    m_kardPlayers.push_back(player2);
    m_kardPlayers.push_back(player3);
    m_kardPlayers.push_back(player4);
    m_kardPlayers.push_back(player5);
    m_kardPlayers.push_back(player6);
    m_kardPlayers.push_back(player7);
    m_playersCards.push_back(player1Cards);
    m_playersCards.push_back(player2Cards);
    m_playersCards.push_back(player3Cards);
    m_playersCards.push_back(player4Cards);
    m_playersCards.push_back(player5Cards);
    m_playersCards.push_back(player6Cards);
    m_playersCards.push_back(player7Cards);

    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player3Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player4Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player5Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player6Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(player7Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player2Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player3Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player4Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player5Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player6Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(player7Cards, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    // Connect the draw pile to the draw slot
    connect(drawPile, SIGNAL(kardSelected(Kard &)), this, SLOT(drawKard(Kard &)));
    // Connect the pass button to the pass slot
    connect(passButton, SIGNAL(clicked()), this, SLOT(passDraw()));
    // Connect the display button to the current score slot
    connect(scores, SIGNAL(clicked()), this, SLOT(displayScores()));

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
}

CrazyEightsInterface::~CrazyEightsInterface()
{}

void CrazyEightsInterface::updateTable()
{
    int size = m_players.size();
    int m_humanPlayerIndex = size - 1;

    // Enable Pass button if the deck is empty, else only show the draw pile
    if (m_deck.isEmpty())
    {
        passButton->setVisible(true);
        passButton->setEnabled(true);
        drawPile->setVisible(false);
    }
    else
    {
        passButton->setVisible(false);
        passButton->setEnabled(false);
        drawPile->setVisible(true);
        drawPile->setCardBackImage(m_pUserProfile->cardBackImageFilename());
        drawPile->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
        drawPile->setCardPile(m_deck);
    }
    discardPile->setCardPile(m_playSequence);
    // Setup the player information
    for (int index = 0; index < size; ++index)
    {
        m_playersCards[index]->setCardBackImage(m_pUserProfile->cardBackImageFilename());
        m_playersCards[index]->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
        m_playersCards[index]->setCardSequence(m_players[index].hand());
        if (index != m_humanPlayerIndex)
            m_kardPlayers[index]->setPlayerImage(m_players[index].name().toLower() + ".png");
        else
        {
            m_kardPlayers[index]->setPlayerImage(m_pUserProfile->userMoodImage("NeutralMood"));
            m_playersCards[index]->setFaceUp(true);
        }
        m_kardPlayers[index]->setDealer(m_players[index].isDealer());
        m_kardPlayers[index]->setTurn(m_players[index].isTurn());
        m_kardPlayers[index]->setSkillLevel(m_players[index].skillLevel());
    }
}

void CrazyEightsInterface::displayMessage(const Player &player, const QString &message)
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
        if (m_players[index].name() == player.name())
        {
            m_kardPlayers[index]->setCaption(message);
            break;
        }
    updateTable();
}

void CrazyEightsInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
    updateTable();
}

Card::Suit CrazyEightsInterface::declareSuit()
{
    SuitRequest suitRequest(QString("Name Play Suit"), false, false, this);

    if (suitRequest.exec())
        return suitRequest.suit();
    else
        return declareSuit();
}

void CrazyEightsInterface::displayRoundSummary(const vector<int> &playerHandPoints, int winningSum)
{
    vector<CardSequence> hands;
    vector<QString> messages;
    int winnerIndex=-1;

    // Show the points the winner has earn form each hand
    for (int index=0; index < m_players.size(); ++index)
        if (playerHandPoints[index] != 0)
        {
            hands.push_back(m_players[index].hand());
            messages.push_back(QString("%1 has a point value of %2.").arg(m_players[index].name()).arg(playerHandPoints[index]));
        }
        else
            winnerIndex=index;
    hands.push_back(m_players[winnerIndex].hand());
    messages.push_back(QString("%1 has won the round, earning %2 points.").arg(m_players[winnerIndex].name()).arg(winningSum));

    KardMessageDisplay messageDisplay(
        "Round Summary",
        hands,
        messages,
        m_pUserProfile->pathToCardFrontImages(),
        m_pUserProfile->cardBackImageFilename(),
        this);
    messageDisplay.exec();
}

void CrazyEightsInterface::gameOver(int playerIndex)
{
    promptMessage(
        "Game Over",
        QString("%1 won with a score of %2.").arg(m_players[playerIndex].name()).arg(m_players[playerIndex].score())
    );
}

void CrazyEightsInterface::roundOverFromBlock()
{
    promptMessage(
        "Round Over",
        QString("No one can play. The round is blocked.\n No one scores.")
    );
}

void CrazyEightsInterface::displayScores()
{
    QString message("Current Standings:\n\n");

    for (int index=0; index < m_players.size(); ++index)
        message+=QString("%1 has %2 points.\n").arg(m_players[index].name()).arg(m_players[index].score());
    promptMessage(
        "Current Scores",
        message
    );
}

void CrazyEightsInterface::playerCardPlayed(Kard &kard)
{
    if (cardPlayed(kard.card()))
        updateTable();
    else
        kard.setSelected(false);
}

void CrazyEightsInterface::playerCardMoved()
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
    {
        m_players[index].hand().clear();
        for (int i = 0, size = m_playersCards[index]->cardSequence().size(); i < size; ++i)
            m_players[index].hand().addCard(m_playersCards[index]->cardSequence()[i]);
    }
}

void CrazyEightsInterface::drawKard(Kard &)
{
    drawCard();
}

void CrazyEightsInterface::passDraw()
{
    passPlay();
}

void CrazyEightsInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (event->timerId() == m_computerTimerId)
        if (m_players.currentPlayer().isTurn() && (m_players.currentPlayerIndex() != (m_players.size() - 1))) // If the current player isn't human computer plays. We check the player's turn because at the end of the game everyone's turn is off.
            computerPlay();
        else
            QWidget::timerEvent(event);
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void CrazyEightsInterface::computerPlay()
{
    int playerIndex = m_players.currentPlayerIndex();
    CrazyEightsPlayer *compPlayer;
    Card cardSelected;

    compPlayer = dynamic_cast<CrazyEightsPlayer *>(&m_players[playerIndex]);
    if (compPlayer == 0) // Human players can't use the computer code.
        throw KardsGTError("CrazyEightsInterface", "computerPlay", "Human player not allowed as a computer player.");
    cardSelected = compPlayer->playCard(m_playSequence);
    qDebug() << "CrazyEightsInterface, computerPlay, Card Selected: " << cardSelected;
    // An empty card indicates a need to draw a card or pass
    if (cardSelected.isEmpty())
        if (m_deck.isEmpty())
            passPlay();
        else
            drawCard();
    else
        m_playersCards[playerIndex]->setSelected(cardSelected);
}
