/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "crazyeights.h"
#include "crazyeightsinterface.h"
#include "kardsgterror.h"
#include "crazyeightsplayer.h"

#include <QFile>
#include <QTextCodec>
#include <QtDebug>

#include <vector>
using std::vector;

CrazyEights::CrazyEights(CrazyEightsInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_deck()
{
    m_pInterface = gameInterface;
}

CrazyEights::~CrazyEights()
{}

void CrazyEights::startPlay()
{
    time_t seconds;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("CrazyEights", "startPlay", "Wrong number of players were added to the game.");
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    m_dealerIndex=rand() % m_players.size();
    // If there is a large group more decks are needed.
    if (m_players.size() > CrazyEightsRules::NUMBER_OF_PLAYERS_FOR_TWO_DECKS)
        m_deck.setNumberOfDecks(CrazyEightsRules::NUMBER_OF_DECKS_FOR_LARGE_GROUP);
    deal();
}

bool CrazyEights::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(QIODevice::WriteOnly))
        return false;
    out.setCodec(QTextCodec::codecForName("UTF-8"));

    out << "Game: " << "crazy_eights" << endl;
    saveGameBase(out);
    out << "NumberOfDecks: " << m_deck.numberOfDecks() << endl;
    out << "Deck: " << m_deck << endl;

    if (file.error() != QFile::NoError)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool CrazyEights::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;
    int temp;

    if (! file.open(QIODevice::ReadOnly))
    {
        file.close();
        return false;
    }
    in.setCodec(QTextCodec::codecForName("UTF-8"));
    in >> label >> label;
    if (label != "crazy_eights")
        return false;
    loadGameBase(in);
    in >> label >> temp;
    m_deck.setNumberOfDecks(temp);
    in >> label >> m_deck;
    file.close();
    m_pInterface->updateTable();
    return true;
}

int CrazyEights::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int CrazyEights::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

bool CrazyEights::cardPlayed(const Card &card)
{
    if (! m_players.currentPlayer().hand().hasCard(card))
    {
        // Find the player who has the card
        for (int index=0; index < m_players.size(); ++index)
            if (m_players[index].hand().hasCard(card))
                m_pInterface->promptMessage("Warning", QString("%1 it is not your turn.").arg(m_players[index].name()));
        return false;
    }

    if (m_rules.isLegalPlay(m_playSequence, card, m_players.currentPlayer()))
    {
        qDebug() << "CrazyEights, Card played: " << card;
        m_players.currentPlayer().hand().removeCard(card);
        m_playSequence.addCardToFront(card); // Added to front to "Discard" to the top of the card pile
        if (card.rank() == Card::EIGHT) // If an eight is played request the suit to change to.
            if (! m_players.currentPlayer().hand().isEmpty()) // Don't ask for suit if the last card played is an eight.
            {
                CrazyEightsPlayer *compPlayer = dynamic_cast<CrazyEightsPlayer *>(& m_players.currentPlayer());

                if (compPlayer == 0)
                    m_playSequence.front().setSuit(m_pInterface->declareSuit());
                else
                    m_playSequence.front().setSuit(compPlayer->selectSuit(m_playSequence));
            }
        m_pInterface->updateTable();
        GameStatusCodes status = static_cast<GameStatusCodes>(handleGameStatus());
        if ((status != ROUND_ENDED) && (status != GAME_OVER))
            setupNextPlayer();
        return true;
    }
    else
    {
        m_pInterface->promptMessage("Warning", QString("%1 is not a legal play.").arg(card.toString()));
        return false;
    }
}

void CrazyEights::deal()
{
    qDebug() << "CrazyEights, Dealing.";
    int cardsToDeal=m_rules.numberOfCardsToDeal(m_players.size());

    clearPlayerHands();
    nextDealer();
    m_deck.shuffle();
    for (int index=0; index < cardsToDeal; ++index)
        for (int playercount = 0; playercount < m_players.size(); ++playercount)
            m_players.nextPlayer().hand().addCard(m_deck.dealCard());
    m_playSequence.addCard(m_deck.dealCard());
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().sortBySuit(); // Give a useful arrangement.
    m_players.setCurrentPlayer(m_players[m_dealerIndex]);
    setupNextPlayer(); // Player to the left of the dealer starts
    m_pInterface->updateTable();
}

void CrazyEights::setupNextPlayer()
{
    qDebug() << "CrazyEights, Setup next player.";
    m_players.currentPlayer().setTurn(false);
    m_players.nextNonEmptyPlayer();
    m_players.currentPlayer().setTurn(true);
}

int CrazyEights::handleGameStatus()
{
    int status=NO_CHANGE;

    if (m_rules.isRoundOver(m_players))
    {
        // Turn off the current player off.
        m_players.currentPlayer().setTurn(false);
        handleEndOfRound();
        status=ROUND_ENDED;
    }
    if (m_rules.isRoundOverFromBlock(m_players, m_deck, m_playSequence))
    {
        // Turn off the current player off.
        m_players.currentPlayer().setTurn(false);
        m_pInterface->roundOverFromBlock();
        status=ROUND_ENDED;
    }
    if (m_rules.isGameOver(m_players))
    {
        // All turns should be off from round check.
        handleEndOfGame();
        status=GAME_OVER;
    }
    qDebug() << "CrazyEights, Handle Game Status: " << status;
    return status;
}

void CrazyEights::drawCard()
{
    qDebug() << "CrazyEights, Draw card.";
    m_players.currentPlayer().hand().addCard(m_deck.dealCard());
    m_pInterface->updateTable();
}

void CrazyEights::passPlay()
{
    qDebug() << "CrazyEights, Pass play.";
    m_pInterface->displayMessage(m_players.currentPlayer(), QString("I pass."));
    setupNextPlayer();
    m_pInterface->updateTable();
}

void CrazyEights::handleEndOfRound()
{
    qDebug() << "CrazyEights, Handling end of round.";
    vector<int> playerHandPoints;
    int points=0;
    int winnerIndex=-1;
    int pointTotal=0;

    // Score the cards
    for (int index=0; index < m_players.size(); ++index)
    {
        points=calculateHandPoints(m_players[index].hand());
        pointTotal+=points;
        if (points == 0)
            winnerIndex=index;
        playerHandPoints.push_back(points);
    }
    m_pInterface->displayRoundSummary(playerHandPoints, pointTotal);
    m_players[winnerIndex].addPoint(pointTotal);
    m_pInterface->displayScores();
    if (! m_rules.isGameOver(m_players))
    {
        m_playSequence.clear();
        deal();
    }
}

int CrazyEights::calculateHandPoints(const CardSequence &hand) const
{
    int points=0;

    for (int index=0; index < hand.size(); ++index)
        points+=m_rules.cardValue(hand[index]);
    return points;
}

void CrazyEights::handleEndOfGame()
{
    qDebug() << "CrazyEights, Handle End Of Game.";
    int winnerIndex=-1;
    int humanIndex=m_players.size() - 1;

    // Find the winner
    for (int index=0, size=m_players.size(); index < size; ++index)
        if (m_players[index].score() >= m_rules.winningGameScore(size))
        {
            winnerIndex=index;
            break;
        }
    m_pInterface->gameOver(winnerIndex);
    // Add the win or loss to the player database.
    if (winnerIndex == humanIndex) // Human player is the last player
        m_pProfileDatabase->setGameStatistics("crazy eights", true);
    else
        m_pProfileDatabase->setGameStatistics("crazy eights", false);
}
