/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef GAMEFACTORY_H
#define GAMEFACTORY_H

#include <vector>
using std::vector;

class QString;
class QStackedWidget;
class GameBase;
class AIBase;
class UserProfileDatabase;

/**
 * This gives us access to all of our games.
 *
 * @author John Schneiderman
 */
class GameFactory
{
public:
    /**
     * General constructor.
     */
    GameFactory();
    /**
     * General destructor.
     */
    ~GameFactory();
    /**
     * Accessor to all of our currently running games.
     * @return our current running game.
     * @throw KardsGTError if you try to access the game before a game has been created.
     */
    const GameBase& game() const;
    /**
     * Accessor to our currently running game.
     * @return our current running game.
     * @throw KardsGTError if you try to access the game before a game has been created.
     */
    GameBase& game();
    /**
     * Creates a card game.
     * @param profileDatabase is the user profile database.
     * @param game is the name of the game we want to create.
     * @note Game names must all be in lowercase.
     * @param layout is the layout to add the game to.
     * @return true if the game was created, false if it was failed to be created.
     */
    bool createGame(UserProfileDatabase &profileDatabase, const QString &game, QStackedWidget &layout);
    /**
     * Clears all of our currently running games.
     */
    void clear();
    /**
     * This tells us if we have a game running.
     * @return true if we have a game, false if we do not.
     */
    bool hasGame() const;

protected:
    /**
     * This setups up a game ready to be played and adds it to our game listing.
     * @param profileDatabase is the user profile database.
     * @param game is the game we want to setup.
     * @param layout is the layout to add the game to.
     * @return true if the game is successfully setup, false if something goes wrong.
     */
    bool setupGame(UserProfileDatabase &profileDatabase, const QString &game, QStackedWidget &layout);

private:
    /// @param m_pGame this is the currently running game.
    GameBase *m_pGame;
};
#endif
