/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kardplayer.h"
#include "kardsgterror.h"

#include <vector>
using std::vector;

#include <QImage>
#include <QEvent>
#include <QPainter>
#include <QPoint>
#include <QFont>
#include <QBrush>
#include <QPixmap>
#include <QTimerEvent>
#include <QPaintEvent>

const QString KardPlayer::CAPTION_MARKER = "~";

KardPlayer::KardPlayer(QWidget *parent): QWidget(parent), m_player(), m_playerImage(), m_caption(), m_captionTimerIds()
{
    m_captionExtended=false;
}

QSize KardPlayer::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const QString& KardPlayer::caption() const
{
    return m_caption;
}

void KardPlayer::setCaption(const QString &caption)
{
    if (m_caption.isEmpty())
        m_caption=caption;
    else
    {
        m_caption=caption + CAPTION_MARKER + m_caption;
        m_captionExtended = true;
    }
    m_captionTimerIds.push(startTimer(CAPTION_DISPLAY_TIME));
    update();
    updateGeometry();
}

void KardPlayer::setCaption(const QString &caption, int length)
{
    if (m_caption.isEmpty())
        m_caption=caption;
    else
    {
        m_caption=caption + CAPTION_MARKER + m_caption;
        m_captionExtended = true;
    }
    m_captionTimerIds.push(startTimer(length));
    update();
    updateGeometry();
}

void KardPlayer::clear()
{
    if (m_caption.isEmpty())
        return;
    if (m_captionExtended)
    {
        int lastCaptionStarts = m_caption.lastIndexOf(CAPTION_MARKER);

        m_caption.remove(lastCaptionStarts, m_caption.length() - lastCaptionStarts);
        // Does our caption still have multiple messages?
        if (m_caption.contains(CAPTION_MARKER) == 0)
            m_captionExtended = false;
    }
    else
        m_caption="";
    killTimer(m_captionTimerIds.top());
    m_captionTimerIds.pop();
    update();
    updateGeometry();
}

void KardPlayer::setDealer(bool dealer)
{
    m_player.setDealer(dealer);
    update();
    updateGeometry();
}

void KardPlayer::setSkillLevel(Player::SkillLevel level)
{
    if ((level > Player::Newb) || (level < Player::Skill_Error))
        throw KardsGTError(QString("KardPlayer"), QString("setSkillLevel"), QString("The value, %1, is an invalid skill level.").arg(level));
    m_player.setSkillLevel(level);
    update();
    updateGeometry();
}

void KardPlayer::setTurn(bool turn)
{
    m_player.setTurn(turn);
    update();
    updateGeometry();
}

QPixmap KardPlayer::playerImage() const
{
    return m_playerImage;
}

void KardPlayer::setPlayerImage(const QPixmap &playerImage)
{
    if (playerImage.isNull())
        throw KardsGTError("KardPlayer", "setPlayerImage", "playerImage is null!");
    else
        m_playerImage=playerImage;
    update();
    updateGeometry();
}

void KardPlayer::setPlayerImage(const QString &imageFilename)
{
    m_playerImage=QPixmap(QString(":/players/" + imageFilename));
    if (m_playerImage.isNull())
    {
        // Try loading it as a path
        m_playerImage=QPixmap(imageFilename);
        if (m_playerImage.isNull())
            throw KardsGTError(QString("KardPlayer"), QString("setPlayerImage"), QString("%1 was not found.").arg(imageFilename));
    }
    update();
    updateGeometry();
}

void KardPlayer::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    painter.setWindow(0, 0, 75, 150);
    draw(painter);
}

void KardPlayer::draw(QPainter &painter)
{
    const int FONT_SIZE = 10; // The font size we want to use.
    const int CHARS_PER_LINE = 7; // How many characters should appear on a single line.
    const int SCALE_FONT_WIDTH = 17; // The scale factor needed to make 1 char = 1 L.U.. Empirically determined.
    const int SCALE_FONT_HEIGHT = 10; // The scale factor needed to make 1 char = 1 L.U.. Empirically determined.
    const int MARGIN = 5; // The caption margin space

    painter.drawPixmap(painter.window(), m_playerImage);
    painter.setFont(QFont("courier", FONT_SIZE));
    if (! m_caption.isEmpty())
    {
        int textX=painter.window().width() / 4; // X-Position for the text
        int textY=painter.window().height() / 4; // Y-Position for the text
        int windowWidth=painter.window().width();
        int width; // Width of our caption bubble
        int height; // height of our caption bubble
        int lines = 1; // How many lines we break up the caption into
        bool textWillFit=true;
        QString caption = formatCaption(CHARS_PER_LINE);

        // Find the width of our caption bubble
        if ((static_cast<int>(m_caption.length()) * SCALE_FONT_WIDTH + textX) < windowWidth)
            width = static_cast<int>(m_caption.length() * SCALE_FONT_WIDTH) + 1;
        else
        {
            width = windowWidth - textX + 1;
            textWillFit = false;
        }
        // Find the height of our caption bubble
        if (textWillFit)
            height = SCALE_FONT_HEIGHT + MARGIN;
        else
        {
            lines = caption.length() / CHARS_PER_LINE;

            if ((caption.length() % CHARS_PER_LINE) != 0)
                ++lines;
            height = (SCALE_FONT_HEIGHT * lines) + MARGIN;
        }

        // Create the caption bubble
        painter.setBackgroundMode(Qt::TransparentMode);
        painter.setBrush(Qt::white);
        painter.drawRect(textX - 1, textY - SCALE_FONT_HEIGHT, width, height);

        // Draw the caption
        painter.setPen(Qt::black);
        if (textWillFit)
            painter.drawText(textX, textY, m_caption);
        else
        {
            int length = caption.length();

            for (int charsPos=0, line = 0; charsPos < length; charsPos+=CHARS_PER_LINE, ++line)
                painter.drawText(textX, textY + (line * SCALE_FONT_HEIGHT), caption.mid(charsPos, CHARS_PER_LINE));
        }
    }
    painter.setPen(Qt::red);
    painter.setBackground(Qt::white);
    painter.setBackgroundMode(Qt::OpaqueMode);
    // Show the player if it's their turn, they're the dealer, else show them the skill level.
    if (m_player.isTurn())
        painter.drawText(0, 10, "T"); // 0, 10 should be the upper-left corner with enough room for the letter.
    else if (m_player.isDealer())
        painter.drawText(0, 10, "D");
    else
        painter.drawText(0, 10, QString("%1").arg(m_player.skillLevel()));
}

void KardPlayer::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == m_captionTimerIds.top())
        clear();
    else
        QWidget::timerEvent(event);
}

QString KardPlayer::formatCaption(int charsPerLine) const
{
    QString formattedCaption = "";
    vector<QString> tokens;

    // Break up the caption into tokens.
    for (int index = 0, size=m_caption.size(), lastPos=0; index < size; ++index)
        if ((m_caption[index] == ' ') || (m_caption[index] == '~'))
        {
            tokens.push_back(m_caption.mid(lastPos, index - lastPos));
            lastPos = index + 1;
        }
        else if ((index + 1) == size)
            tokens.push_back(m_caption.mid(lastPos, index - lastPos + 1));
    for (int index = 0, size = tokens.size(), currentSize = 0; index < size; ++index)
    {
        if (currentSize == charsPerLine) // If we are equal to the max, reset.
            currentSize = 0;
        if ((currentSize + tokens[index].size() + 1) <= charsPerLine) // Token can fit on the line with a space.
        {
            formattedCaption += tokens[index] + " ";
            currentSize += tokens[index].size() + 1;
        }
        else if ((currentSize + tokens[index].size()) <= charsPerLine) // Token goes up to the max line.
        {
            formattedCaption += tokens[index];
            currentSize += tokens[index].size();
            currentSize = 0;
        }
        else if (tokens[index].size() > charsPerLine) // Token can not fit on the max line, split token up.
        {
            int currentPos=0;
            int tokenSize=tokens[index].size();

            formattedCaption += tokens[index].mid(0, charsPerLine - currentSize - 1);
            formattedCaption += "-";
            currentPos = charsPerLine - currentSize - 1;
            currentSize = 0;
            while (currentPos < tokenSize)
            {
                if ((currentPos + charsPerLine) < tokenSize)
                {
                    formattedCaption += tokens[index].mid(currentPos, charsPerLine - 1);
                    formattedCaption += "-";
                    --currentPos;
                    currentSize = 0;
                }
                else
                {
                    formattedCaption += tokens[index].mid(currentPos, charsPerLine);
                    currentSize += (tokenSize - currentPos);
                }
                currentPos += charsPerLine;
                formattedCaption += " ";
                if ((currentSize + 1) == charsPerLine)
                    currentSize = 0;
                else
                    ++currentSize;
            }
        }
        else // Token needs to move to the next "line", by adding spaces.
        {
            int padding=charsPerLine - currentSize;

            for (int padIndex=0; padIndex < padding; ++padIndex)
                formattedCaption+=" ";
            formattedCaption += tokens[index];
            currentSize = tokens[index].size();
            if ((currentSize + 1) <= charsPerLine)
            {
                formattedCaption += " ";
                ++currentSize;
            }
        }
    }
    return formattedCaption;
}
