/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kard.h"
#include "kardsgterror.h"

#include <QPainter>
#include <QImage>
#include <QEvent>
#include <QRect>
#include <QPixmap>
#include <QMouseEvent>
#include <QPaintEvent>
#include <QDrag>
#include <QApplication>

Kard::Kard(QWidget *parent): QWidget(parent), m_cardFront(), m_card()
{
    m_faceUp=m_selected=false;
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_cardBack = QPixmap(":/cardBacks/back00.png");
    m_cardPath = "";
}

Kard::Kard(const Card &card, bool faceUp, QWidget *parent): QWidget(parent), m_cardFront()
{
    m_selected=false;
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_cardBack = QPixmap(":/cardBacks/back00.png");
    setCard(card);
    setFaceUp(faceUp);
    m_cardPath = "";
}

Kard::~Kard()
{}

QSize Kard::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const QPixmap& Kard::cardBack() const
{
    return m_cardBack;
}

const QPixmap& Kard::cardFront() const
{
    return m_cardFront;
}

bool Kard::isFaceUp() const
{
    return m_faceUp;
}

bool Kard::isSelected() const
{
    return m_selected;
}

void Kard::setCardBack(const QPixmap &cardBack)
{
    if (cardBack.isNull())
        throw KardsGTError("Kard", "setCardBack", "cardBack is null!");
    else
        m_cardBack=cardBack;
    update();
    updateGeometry();
}

void Kard::setCardBack(const QString &cardBackFilename)
{
    // If we pass in a null string use the default
    if (cardBackFilename.isEmpty())
        m_cardBack = QPixmap(":/cardBacks/back00.png");
    else
        m_cardBack = QPixmap(QString(":/cardBacks/" + cardBackFilename)); // Is it our built in card types
    // Is it an external card image
    if (m_cardBack.isNull())
    {
        m_cardBack = QPixmap(cardBackFilename);
        if (cardBackFilename.isEmpty())
            throw KardsGTError(QString("Kard"), QString("setCardBack"), QString("cardBackFilename %1 was not found!").arg(cardBackFilename));
    }
    update();
    updateGeometry();
}

void Kard::setCardFront(const QPixmap &cardFront)
{
    if (cardFront.isNull())
        throw KardsGTError("Kard", "setCardFront", "cardFront is null!");
    else
        m_cardFront=cardFront;
    update();
    updateGeometry();
}

void Kard::setCardPath(const QString &cardPath)
{
    m_cardPath = cardPath;
    setCard(m_card);
}

void Kard::setCard(const Card &card)
{
    m_card = card;
    if (! card.isEmpty())
    {
        if ((m_cardPath == "default") || m_cardPath.isEmpty())
        {
            m_cardFront = QPixmap(QString(":/cardFronts/" + card.toString().toLower() + ".xpm"));
            if (m_cardFront.isNull()) // See if the image is a PNG file.
                m_cardFront = QPixmap(QString(":/cardFronts/" + card.toString().toLower() + ".png"));
        }
        else
        {
            m_cardFront = QPixmap(QString(m_cardPath + card.toString().toLower() + ".xpm"));
            if (m_cardFront.isNull()) // See if the image is a PNG file.
                m_cardFront = QPixmap(QString(m_cardPath + card.toString().toLower() + ".png"));
        }
        if (m_cardFront.isNull())
            throw KardsGTError(QString("Kard"), QString("setCard"), QString("%1.xpm|png image not found!").arg(m_cardPath + card.toString().toLower()));
    }
    else
        m_cardFront.fill();
    update();
    updateGeometry();
}

void Kard::setFaceUp(bool faceUp)
{
    m_faceUp=faceUp;
    update();
    updateGeometry();
}

void Kard::setSelected(bool select)
{
    // Make sure we only emit the signal once.
    if (m_selected == select)
        return;

    m_selected=select;
    if (m_selected)
        emit cardSelected(*this);
}

void Kard::flipCard()
{
    if (m_faceUp)
        m_faceUp=false;
    else
        m_faceUp=true;
    update();
    updateGeometry();
}

void Kard::draw(QPainter &painter)
{
    //Draw a plain rectangle if nothing is set, else draw either the front of the card or the back of the card.
    if (m_card.isEmpty())
        painter.drawRect(this->rect());
    else if (m_faceUp)
        painter.drawPixmap(QRect(0, 0, PREFERRED_WIDTH, PREFERRED_HEIGHT), m_cardFront);
    else
        painter.drawPixmap(QRect(0, 0, PREFERRED_WIDTH, PREFERRED_HEIGHT), m_cardBack);
}

Kard& Kard::operator=(const Kard &kard)
{
    m_cardBack=kard.m_cardBack;
    m_cardFront=kard.m_cardFront;
    m_faceUp=kard.m_faceUp;
    m_selected=kard.m_selected;
    m_card=kard.m_card;
    update();
    updateGeometry();
    return *this;
}

void Kard::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    draw(painter);
}

void Kard::mouseDoubleClickEvent(QMouseEvent *event)
{
    // Prevent a card from being selected more than once.
    if (m_selected)
        return;
    // User selected a card.
    if (event->button() == Qt::LeftButton)
        setSelected(true);
}

void Kard::mousePressEvent(QMouseEvent *event)
{
    // Drag our kard.
    if (event->button() == Qt::LeftButton)
        m_dragPosition = event->pos();
    else
        QWidget::mousePressEvent(event);
}

void Kard::mouseMoveEvent(QMouseEvent *event)
{
    // Are we dragging our kard with the left mouse button?
    if (event->buttons() & Qt::LeftButton)
    {
        int distance = (event->pos() - m_dragPosition).manhattanLength();

        if (distance >= QApplication::startDragDistance())
            startDrag();
    }
}

void Kard::startDrag()
{
    QDrag *drag = new QDrag(this->parentWidget());
    QMimeData *mimeData = new QMimeData();
    QByteArray rawKard(m_card.toString().toStdString().c_str());

    mimeData->setData("application/x-kardsgt.text.kard", rawKard);
    drag->setMimeData(mimeData);
    if (m_faceUp)
        drag->setPixmap(m_cardFront.scaled(100, 100, Qt::KeepAspectRatio));
    else
        drag->setPixmap(m_cardBack.scaled(100, 100, Qt::KeepAspectRatio));
    drag->exec();
}

const Card& Kard::card()
{
    return m_card;
}
