/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "carddeck.h"

#include <QString>

#include <algorithm>
#include <ctime>
#include <vector>
using std::vector;
using std::pair;

CardDeck::CardDeck(int numberOfDecks): CardPile(), m_removedCards()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    m_numberOfDecks = numberOfDecks;
    m_numberOfJokers = 0;
    createDeck();
}

CardDeck::CardDeck(vector<Card> cards, int numberOfDecks): CardPile()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    m_removedCards = cards;
    m_numberOfDecks = numberOfDecks;
    m_numberOfJokers = 0;
    createDeck();
}

void CardDeck::shuffle()
{
    // See if the deck is missing some cards
    if (m_pile.size() != m_deckSize)
    {
        CardSequence missingCards = findMissingCards();

        // Shuffle the missing cards, because no one sorts the cards before they put them back in the deck
        random_shuffle(missingCards.begin(), missingCards.end());
        // Add any missing cards
        for (int index=0; index < missingCards.size(); ++index)
            m_pile.addCard(missingCards[index]);
    }

    // Shuffle the deck
    random_shuffle(m_pile.begin(), m_pile.end());
}

Card CardDeck::dealCard()
{
    return removeCard();
}

void CardDeck::cutDeck()
{
    // Ensure that we at least select GUARANTEE_CUT_SIZE cards.
    int newTopCardIndex=(rand() % (m_pile.size() - (GUARANTEE_CUT_SIZE * m_numberOfDecks))) + (GUARANTEE_CUT_SIZE * m_numberOfDecks);
    int pileSize=m_pile.size();
    CardSequence leftDeck, rightDeck;

    // Cut the deck
    for (int cardIndex=0; cardIndex < pileSize; cardIndex++)
        if (cardIndex < newTopCardIndex)
            leftDeck.addCard(m_pile[cardIndex]);
        else
            rightDeck.addCard(m_pile[cardIndex]);

    // Combine the two decks.
    m_pile = rightDeck;
    m_pile += leftDeck;
}

void CardDeck::createDeck()
{
    Card card;

    m_deckSize = (((Card::NUMBER_OF_SUITS * Card::NUMBER_OF_RANKS) + m_numberOfJokers - m_removedCards.size()) * m_numberOfDecks);
    m_pile.clear();
    for (int deckCount = 0; deckCount < m_numberOfDecks; ++deckCount)
        for (int suit=0; suit < Card::NUMBER_OF_SUITS; ++suit)
            for (int rank=0; rank < Card::NUMBER_OF_RANKS; ++rank)
            {
                card = Card(static_cast<Card::Rank>(rank), static_cast<Card::Suit>(suit));
                if (allowedCard(card))
                    m_pile.addCard(card);
            }
    insertJokers(m_pile);
}

CardSequence CardDeck::findMissingCards() const
{
    CardSequence missingCards;
    vector<pair<Card, int> > numberOfCard; // First is the card to track, second is the number of that card in the deck
    Card card;

    // Initialise vector with a deck of cards
    for (int suit=0; suit < Card::NUMBER_OF_SUITS; ++suit)
        for (int rank=0; rank < Card::NUMBER_OF_RANKS; ++rank)
        {
            card = Card(static_cast<Card::Rank>(rank), static_cast<Card::Suit>(suit));
                numberOfCard.push_back(pair<Card, int>(card, 0));
        }
    // Count up each of the cards
    for (int index = 0, size = m_pile.size(), cardIndex; index < size; ++index)
    {
        cardIndex = m_pile[index].rank() + (m_pile[index].suit() * Card::NUMBER_OF_RANKS);
        numberOfCard[cardIndex].second++;
    }
    // Find the missing cards
    for (int index = 0, size = numberOfCard.size(); index < size; ++index)
        if (allowedCard(numberOfCard[index].first))
            while (numberOfCard[index].second < m_numberOfDecks)
            {
                missingCards.addCard(numberOfCard[index].first);
                numberOfCard[index].second++;
            }
    insertJokers(missingCards);
    return missingCards;
}

void CardDeck::removeCards(Card::Rank rank)
{
    Card card;

    for (int suit=0; suit < Card::NUMBER_OF_SUITS; ++suit)
    {
        card = Card(rank, static_cast<Card::Suit>(suit));
        m_removedCards.push_back(card);
    }
    createDeck();
}

void CardDeck::removeCards(const Card &card)
{
    m_removedCards.push_back(card);
    createDeck();
}

void CardDeck::addJoker()
{
    m_numberOfJokers++;
    createDeck();
}

bool CardDeck::allowedCard(const Card &card) const
{
    for (int index = 0, size = m_removedCards.size(); index < size; ++index)
        if (card == m_removedCards[index])
            return false;
    return true;
}

int CardDeck::numberOfDecks() const
{
    return m_numberOfDecks;
}

void CardDeck::setNumberOfDecks(int numberOfDecks)
{
    m_numberOfDecks = numberOfDecks;
    createDeck();
}

void CardDeck::insertJokers(CardSequence &pile) const
{
    int numberOfJokersInPile = 0;
    Card joker(Card::JOKER, Card::SUIT_ERR);

    // Count the number of jokers currently in the deck.
    for (int index = 0, size = m_pile.size(); index < size; ++index)
        if (m_pile[index] == joker)
            ++numberOfJokersInPile;
    // Add the missing jokers
    for (int jokersToAdd = (m_numberOfJokers * m_numberOfDecks) - numberOfJokersInPile; jokersToAdd < m_numberOfJokers; ++jokersToAdd)
        pile.addCard(joker);
}
