/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "wally.h"
#include "card.h"
#include "cardsequence.h"
#include "rulebase.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// Games Wally knows how to play.
#include "wallycrazyeightsai.h"
#include "wallycribbageai.h"
#include "wallyheartsai.h"
#include "wallyspadesai.h"
#include "wallywarai.h"
#include "wallyeuchreai.h"

#include <QString>

Wally::Wally()
{}

Wally::~Wally()
{}

QString Wally::name() const
{
    return "Wally";
}

CardSequence Wally::whichCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else if (game.contains("euchre"))
        cards = handleEuchreCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("crazy eights"))
        cards = handleCrazyEightsCards(game, playSequence, rules, hand, specialInformation);
    else if (game == "war")
    {
        WallyWarAI ai(hand);
        cards=ai.selectCards();
    }
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Wally::decision(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else if (game.contains("euchre"))
        bid = handleEuchreDecisions(game, playSequence, rules, hand, specialInformation, score);
    else if (game.contains("crazy eights"))
        bid = handleCrazyEightsDecisions(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::SkillLevel Wally::skillLevel(const QString &game) const
{
    if (game == "cribbage")
        return Player::Professional;
    else if (game == "spades")
        return Player::Apprentice;
    else if (game == "war")
        return Player::Master;
    else if (game == "hearts")
        return Player::Novice;
    else if (game == "old maid")
        return Player::Master;
    else if (game == "euchre")
        return Player::Novice;
    else if (game == "crazy eights")
        return Player::Amateur;
    else
        return Player::Newb;
}

CardSequence Wally::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        WallyCribbageAI ai(playSequence, rules, hand);
        return ai.selectCards();
    }
    else if (game == "cribbageDealerCrib")
    {
        WallyCribbageAI ai(playSequence, rules, hand);
        return ai.selectDealersCribCards();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        WallyCribbageAI ai(playSequence, rules, hand);
        return ai.selectNonDealersCribCards();
    }
    throw KardsGTError("Wally", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Wally::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        WallySpadesAI ai(playSequence, rules, hand);
        Card partnersCard;

        partnersCard.setCard(specialInformation.value("Partner Played"));
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards(partnersCard);
    }
    else if (game == "spadesExchangeCards")
    {
        WallySpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Wally", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Wally::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        WallyHeartsAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "heartsPassPhase")
    {
        WallyHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Wally", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Wally::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        WallySpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        WallySpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Wally", "handleSpadesBid", "Failed to determine game type or phase.");
}

int Wally::handleEuchreDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "euchreMakeTurnUp")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.turnUp(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreSelectTrump")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.selectTrump(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchrePlayAlone")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.playAlone(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreDefendAlone")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.defendAlone();
    }
    else if (game == "euchreWantsTurnUp")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.wantTurnUp();
    }
    throw KardsGTError("Wally", "handleEuchreDecisions", "Failed to determine game type or phase.");
}

CardSequence Wally::handleEuchreCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "euchre")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        Card partnersCard;

        partnersCard.setCard(specialInformation.value("Partner Played"));
        return ai.selectCards(partnersCard);
    }
    else if (game == "euchreDiscard")
    {
        WallyEuchreAI ai(playSequence, rules, hand);
        return ai.discard();
    }
    throw KardsGTError("Wally", "handleEuchreCards", "Failed to determine game type or phase.");
}

CardSequence Wally::handleCrazyEightsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "crazy eights SelectCard")
    {
        WallyCrazyEightsAI ai(playSequence, rules, hand, specialInformation);
        return ai.selectCard();
    }
    throw KardsGTError("Wally", "handleCrazyEightsCards", "Failed to determine game type or phase.");
}

int Wally::handleCrazyEightsDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "crazy eights SelectSuit")
    {
        WallyCrazyEightsAI ai(playSequence, rules, hand, specialInformation);
        return ai.selectSuit(score);
    }
    throw KardsGTError("Wally", "handleCrazyEightsDecisions", "Failed to determine game type or phase.");
}
