/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef ABIGAIL_H
#define ABIGAIL_H

#include "aibase.h"

/**
 * This is the character strategies for Abigail.
 *
 * @author John Schneiderman
 */
class Abigail: public AIBase
{
public:
    /**
     * The default constructor.
     */
    Abigail();
    /**
     * The default destructor.
     */
    ~Abigail();
    /**
     * This selects cards from her hand.
     * @param game is the game or game phase she's playing.
     * @param playSequence is the current play sequence.
     * @param rules are the rules for the game.
     * @param hand is her current hand.
     * @param specialInformation is the game state information provided from the game.
     * @return the cards she selects, or if game cannot be determined a random card from hand.
     */
    virtual CardSequence whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const;
    /**
     * This makes a decision based on the cards in her hand.
     * @param game is the game or game phase she's playing.
     * @param playSequence is the current play sequence.
     * @param rules are the rules for the game.
     * @param hand is her current hand.
     * @param specialInformation is the game state information provided from the game.
     * @param score is our current score in the game.
     * @return the decision she desires, or if game cannot be determined a random number [1, 100].
     */
    virtual int decision(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const;
    /**
     * This gives access to the skill level strategies for a specific game.
     * @param game is the game we want to know her skill level of.
     * @return her skill level.
     */
    virtual Player::SkillLevel skillLevel(const QString& game) const;
    /**
     * This gives her name.
     * @return her name.
     */
    virtual QString name() const;

private:
    // Card Handling methods
    /**
     * Handles the cribbage ai.
     * @param game is the phase of the game we're in.
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @return the cards we've selected.
     * @throw KardsGTError if we cannot determine what to do with game.
     */
    CardSequence handleCribbageCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const;
    /**
     * Handles the hearts ai.
     * @param game is the phase of the game we're in.
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @return the cards we've selected.
     * @throw KardsGTError if we cannot determine what to do with game.
     */
    CardSequence handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const;
    /**
     * Handles the spades ai.
     * @param game is the phase of the game we're in.
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @return the cards we've selected.
     * @throw KardsGTError if we cannot determine what to do with game.
     */
    CardSequence handleSpadesCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const;
    /**
     * Handles the euchre ai.
     * @param game is the phase of the game we're in.
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @return the cards we've selected.
     * @throw KardsGTError if we cannot determine what to do with game.
     */
    CardSequence handleEuchreCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const;
    /**
     * Handles the crazy eights ai.
     * @param game is the phase of the game we're in.
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @return the cards we've selected.
     * @throw KardsGTError if we cannot determine what to do with game.
     */
    CardSequence handleCrazyEightsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const;
    // Bid Handling methods
    /**
     * Handles the bidding ai for crazy eights.
     * @param game is the phase of the game we're
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @param score is our current score in the game.
     * @return our decision.
     */
    int handleCrazyEightsDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const;
    /**
     * Handles the bidding ai for spades.
     * @param game is the phase of the game we're
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @param score is our current score in the game.
     * @return our bid.
     */
    int handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const;
    /**
     * Handles the bidding ai for euchre.
     * @param game is the phase of the game we're
     * @param playSequence is the current play sequence.
     * @param rules is the rules for our game.
     * @param hand is our hand.
     * @param specialInformation is the game state information provided from the game.
     * @param score is our current score in the game.
     * @return our decision.
     */
    int handleEuchreDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const;
};
#endif
