/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kardsgtinterface.h"
#include "kardsgterror.h"

#include <QApplication>

#include <iostream>
using std::cout;
using std::cerr;
using std::endl;
#include <string>
using std::string;

/// @param VERSION is the current version of our programme.
#ifdef QT_NO_DEBUG
    extern const string VERSION="0.7.0";
#else
    extern const string VERSION="0.7.0 DEBUG";
#endif
/// @param COPYRIGHT is the current copyright years of our programme.
extern const string COPYRIGHT_YEARS="2005 - 2008";

/**
 * This is the entry to start our programme.
 *
 * @param argc is the number of arguments given at the command line.
 * @param argv is the array containing the arguments. The arguments available are: \n
 * -v for the version information. \n
 * -h for a short help information. \n
 * -l game  This will load a game file, where game is a KardsGT game file.
 * @return whether the programme exited successfully or not.
 */
int main(int argc, char * argv[])
{
    // Launch the game.
    try
    {
        KardsGTInterface *mainWindow;
        string gameToLoad="";
        QCoreApplication::setOrganizationName("CodeGNU");
        QCoreApplication::setOrganizationDomain("kardsgt.nongnu.org");
        QCoreApplication::setApplicationName("KardsGT");
        QApplication application(argc, argv);

        // Process our argument list
        for (int index=1, size=argc; index < size; ++index)
        {
            if (strcmp(argv[index], "-h") == 0)
            {
                cout << "usage: kardsgt [options]" << endl;
                cout << "Options:" << endl;
                cout << "  -v                       Shows the version information." << endl;
                cout << "  -l <game>                Loads a game KardsGT game." << endl;
                cout << endl << "For more information see the man page." << endl;
                cout << "For bug reporting instructions, please see the man page." << endl;
                return EXIT_SUCCESS;
            }
            else if (strcmp(argv[index], "-l") == 0)
            {
                if ((index + 1) < argc) // Ensure that a filename was given
                    gameToLoad=argv[++index];
                else
                {
                    cout << "Failed to specify game file to load!" << endl;
                    return EXIT_FAILURE;
                }
            }
            else if (strcmp(argv[index], "-v") == 0)
            {
                cout << "KardsGT version " << VERSION << endl;
                cout << "Copyright (C) "<< COPYRIGHT_YEARS << " John Schneiderman" << endl;
                return EXIT_SUCCESS;
            }
            else
            {
                cerr << "Unrecognised option encountered " << argv[index] << endl;
                return EXIT_FAILURE;
            }
        }
        Q_INIT_RESOURCE(images);
        mainWindow=new KardsGTInterface(QString(gameToLoad.c_str()));
        mainWindow->show();
        application.connect(&application, SIGNAL(lastWindowClosed()), &application, SLOT(quit()));
        // Display our copyright notice.
        cout << "KardsGT version " << VERSION << endl;
        cout << "Copyright (C) "<< COPYRIGHT_YEARS << " John Schneiderman" << endl;
        cout << "KardsGT comes with ABSOLUTELY NO WARRANTY." << endl;
        cout << "This is free software, and you are welcome to redistribute it" << endl;
        cout << "under certain conditions; see the COPYING file for details," << endl;
        cout << "or the Free Software Foundation's GPL." << endl << endl;
        return application.exec();
    }
    catch (KardsGTError error)
    {
        cerr << error << endl;
        return EXIT_FAILURE;
    }
    catch (...)
    {
        cerr << "Unknown error occurred..." << endl;
        return EXIT_FAILURE;
    }
}
