/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2007-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "oldmaid.h"
#include "oldmaidinterface.h"
#include "kardsgterror.h"
#include "oldmaidplayer.h"

#include <QFile>
#include <QTextStream>
#include <QTextCodec>

OldMaid::OldMaid(OldMaidInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_deck()
{
    m_pInterface = gameInterface;
    m_humanPlayerIndex = -1;
    m_deck.removeCards(Card(Card::QUEEN, Card::CLUBS)); // Permanently remove one Queen.
}

OldMaid::~OldMaid()
{}

void OldMaid::startPlay()
{
    m_humanPlayerIndex = m_players.size() - 1;
    time_t seconds;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("OldMaid", "startPlay", "Wrong number of players were added to the game.");
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    m_dealerIndex=rand() % m_players.size();
    deal();
}

bool OldMaid::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(QIODevice::WriteOnly))
        return false;
    out.setCodec(QTextCodec::codecForName("UTF-8"));

    out << "Game: " << "old_maid" << endl;
    // Begin GameBase Save
    saveGameBase(out);
    // End GameBase Save
    out << "Deck: " << m_deck << endl;
    out << "HumanPlayerIndex: " << m_humanPlayerIndex << endl;
    if (file.error() != QFile::NoError)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool OldMaid::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;

    if (! file.open(QIODevice::ReadOnly))
    {
        file.close();
        return false;
    }
    in.setCodec(QTextCodec::codecForName("UTF-8"));
    in >> label >> label;
    if (label != "old_maid")
        return false;
    // Begin GameBase Load
    loadGameBase(in);
    // End GameBase Load
    in >> label >> m_deck;
    in >> label >> m_humanPlayerIndex;
    file.close();
    m_pInterface->updateTable();
    return true;
}

int OldMaid::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int OldMaid::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

bool OldMaid::cardPlayed(const Card &card)
{
    if (m_players.leftPlayerWithCards().hand().hasCard(card))
    {
        if (m_rules.isLegalPlay(m_players.leftPlayerWithCards().hand(), card, m_players.currentPlayer()))
        {
            m_players.currentPlayer().hand().addCard(card);
            m_players.leftPlayerWithCards().hand().removeCard(card);
            m_pInterface->updateTable();
            handlePairs();
            // Give the Computer player the opertunity to shuffle his cards
            OldMaidPlayer *compPlayer = dynamic_cast<OldMaidPlayer *>(& m_players.currentPlayer());
            if (compPlayer != 0)
                compPlayer->rearrangeCards();
            m_pInterface->updateTable();
            GameStatusCodes status = static_cast<GameStatusCodes>(handleGameStatus());
            if (status != OldMaid::GAME_OVER)
                setupNextPlayer();
            else
            {
                for (int i=0; i < m_players.size(); ++i)
                    m_players[i].setTurn(false);
                m_pInterface->reset();
                deal();
            }
        }
        return false;
    }
    else
        throw KardsGTError(QString("OldMaid"), QString("cardPlayed"), QString("Failed to find card %1").arg(card.toString()));
}

void OldMaid::deal()
{
    for (int i = 0; i < m_players.size(); ++i)
        m_players[i].hand().clear();
    nextDealer();
    m_deck.shuffle();
    while (! m_deck.isEmpty()) // Deal out all the cards
        m_players.nextPlayer().hand().addCard(m_deck.dealCard());
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().sortBySuit(); // Give a usefull arrangement.
    m_pInterface->updateTable(); // So the human player can see who is where.
    handlePairs();
    setupNextPlayer();
    m_pInterface->updateTable();
}

void OldMaid::setupNextPlayer()
{
    if (! m_rules.isGameOver(m_players)) // See if the game is over before we start someones turn.
    {
        m_players.currentPlayer().setTurn(false);
        m_players.nextNonEmptyPlayer();
        m_players.currentPlayer().setTurn(true);
    }
}

int OldMaid::handleGameStatus()
{
    if (m_rules.isGameOver(m_players))
    {
        m_pInterface->gameWon();
        return GAME_OVER;
    }
    else
        return NORMAL;
}

int OldMaid::leftPlayerIndex() const
{
    int currentPlayerIndex = 0;
    int size = m_players.size();
    int leftPlayerIndex = -1;

    // Find the current player's index
    for (int index = 0; index < size; ++index)
        if (m_players[index].isTurn())
        {
            currentPlayerIndex = index;
            break;
        }
    leftPlayerIndex = (currentPlayerIndex + 1) % size;

    while (m_players[leftPlayerIndex].hand().isEmpty())
        leftPlayerIndex = (leftPlayerIndex + 1) % size;
    if (leftPlayerIndex == currentPlayerIndex)
        throw KardsGTError("OldMaid", "leftPlayerIndex", "Failed to find a left player with cards!");
    return leftPlayerIndex;
}

void OldMaid::handlePairs()
{
    OldMaidPlayer *compPlayer = NULL;
    CardSequence pairs;

    m_players.currentPlayer().setTurn(false);
    for (int index = 0; index < m_players.size(); ++index)
    {
        compPlayer = dynamic_cast<OldMaidPlayer *>(& m_players[index]);

        if (compPlayer == 0) // Human player
        {
            pairs = m_pInterface->discardPairs();

            while (pairs.size() != numberOfPairs(m_players[index].hand()))
                pairs = m_pInterface->discardPairs();
            m_players[index].hand() -= pairs;
        }
        else
        {
            pairs = compPlayer->discardPairs();
            m_players[index].hand() -= pairs;
            if (! pairs.isEmpty())
                m_pInterface->showDiscardedPairs(m_players[index].name(), pairs);
        }
    }
    m_players.currentPlayer().setTurn(true);
}

int OldMaid::numberOfPairs(CardSequence seq) const
{
    CardSequence pairs;
    CardSequence hand = seq;
    int size = seq.size() - 1;

    sort(hand.begin(), hand.end());
    for (int index=0; index < size; ++index)
        if (m_rules.rankValue(hand[index]) == m_rules.rankValue(hand[index + 1]))
        {
            pairs.addCard(hand[index]);
            pairs.addCard(hand[index + 1]);
            ++index;
        }
    return pairs.size();
}
