/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "heartsinterface.h"
#include "kardsequence.h"
#include "kardplayer.h"
#include "cardsequence.h"
#include "player.h"
#include "kardselection.h"
#include "kard.h"
#include "heartsplayer.h"
#include "kardsgterror.h"
#include "kardmessagedisplay.h"
#include "userprofiledatabase.h"
#include "kardplaysequence.h"

#include <QLCDNumber>
#include <QMessageBox>
#include <QTimerEvent>
#include <QColor>
#include <QPalette>

HeartsInterface::HeartsInterface(UserProfileDatabase &profileDatabase, QWidget *parent): QWidget(parent), Hearts(this, profileDatabase)
{
    Ui::HeartsInterface::setupUi(this);
    // signals and slots connections
    connect(bttnDisplayScores, SIGNAL(clicked()), this, SLOT(displayCurrentScores()));
    connect(bttnDisplayTricksTaken, SIGNAL(clicked()), this, SLOT(displayTricksTaken()));
    m_pUserProfile = &profileDatabase;

    //Set the colours for our game
    setBackgroundRole(QPalette::Background);
    setAutoFillBackground (true);
    QPalette pal(this->palette());
    pal.setColor(QPalette::Background, Qt::darkGreen);
    this->setPalette(pal);

    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
    connect(player3Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player3CardPlayed(Kard &)));
    connect(player4Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player4CardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved()), this, SLOT(player1CardMoved()));
    connect(player2Cards, SIGNAL(kardMoved()), this, SLOT(player2CardMoved()));
    connect(player3Cards, SIGNAL(kardMoved()), this, SLOT(player3CardMoved()));
    connect(player4Cards, SIGNAL(kardMoved()), this, SLOT(player4CardMoved()));

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
}

HeartsInterface::~HeartsInterface()
{}

void HeartsInterface::updateTable()
{
    // Set the card back images
    player1Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    player2Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    player3Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    player4Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    discardPile->setCardBackImage(m_pUserProfile->cardBackImageFilename());

    // Set the card front path
    player1Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    player2Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    player3Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    player4Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    discardPile->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());

    // Update player hands
    player1Cards->setCardSequence(m_players[Hearts::PLAYER_ONE_INDEX].hand());
    player2Cards->setCardSequence(m_players[Hearts::PLAYER_TWO_INDEX].hand());
    player3Cards->setCardSequence(m_players[Hearts::PLAYER_THREE_INDEX].hand());
    player4Cards->setCardSequence(m_players[Hearts::PLAYER_FOUR_INDEX].hand());
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        discardPile->setCardSequence(m_playSequence, m_playerCardOrderIndexes.front());

    // Update table info
    // Player 1
    player1->setPlayerImage(m_players[Hearts::PLAYER_ONE_INDEX].name().toLower() + ".png");
    player1->setDealer(m_players[Hearts::PLAYER_ONE_INDEX].isDealer());
    player1->setTurn(m_players[Hearts::PLAYER_ONE_INDEX].isTurn());
    player1->setSkillLevel(m_players[Hearts::PLAYER_ONE_INDEX].skillLevel());
    // player 2
    player2->setPlayerImage(m_players[Hearts::PLAYER_TWO_INDEX].name().toLower() + ".png");
    player2->setDealer(m_players[Hearts::PLAYER_TWO_INDEX].isDealer());
    player2->setTurn(m_players[Hearts::PLAYER_TWO_INDEX].isTurn());
    player2->setSkillLevel(m_players[Hearts::PLAYER_TWO_INDEX].skillLevel());
    // player 3
    player3->setPlayerImage(m_pUserProfile->userMoodImage("NeutralMood"));
    player3->setDealer(m_players[Hearts::PLAYER_THREE_INDEX].isDealer());
    player3->setTurn(m_players[Hearts::PLAYER_THREE_INDEX].isTurn());
    player3->setSkillLevel(m_players[Hearts::PLAYER_THREE_INDEX].skillLevel());
    // player 4
    player4->setPlayerImage(m_players[Hearts::PLAYER_FOUR_INDEX].name().toLower() + ".png");
    player4->setDealer(m_players[Hearts::PLAYER_FOUR_INDEX].isDealer());
    player4->setTurn(m_players[Hearts::PLAYER_FOUR_INDEX].isTurn());
    player4->setSkillLevel(m_players[Hearts::PLAYER_FOUR_INDEX].skillLevel());
}

CardSequence HeartsInterface::passCards(const Player &player, const QString &direction)
{
    KardSelection passSelection(QString("Select %1 cards to pass %2.").arg(HeartsRules::PASS_TO_PLAYER).arg(direction), HeartsRules::PASS_TO_PLAYER, player.hand(), m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);

    if (passSelection.exec())
        return passSelection.selection();
    else
        return passCards(player, direction);
}

void HeartsInterface::displayMessage(const Player &player, const QString &message)
{
    if (player.name() == m_players[Hearts::PLAYER_ONE_INDEX].name())
        player1->setCaption(message);
    else if (player.name() == m_players[Hearts::PLAYER_TWO_INDEX].name())
        player2->setCaption(message);
    else if (player.name() == m_players[Hearts::PLAYER_THREE_INDEX].name())
        player3->setCaption(message);
    else if (player.name() == m_players[Hearts::PLAYER_FOUR_INDEX].name())
        player4->setCaption(message);
    else
        throw KardsGTError("HeartsInterface", "displayMessage", "Player not found!");
}

void HeartsInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
}

void HeartsInterface::displayRoundSummary(const QString &caption, const vector<CardSequence> &displayHands, const vector<QString> &messages)
{
    KardMessageDisplay summary(caption, displayHands, messages, m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);

    summary.exec();
    updateTable();
}

void HeartsInterface::displayCurrentScores()
{
    int size=m_players.size();

    for (int playerIndex=0; playerIndex < size; ++playerIndex)
        displayMessage(m_players[playerIndex], QString("I have %1 points.").arg(m_players[playerIndex].score()));
}

void HeartsInterface::displayTricksTaken()
{
    int size=m_players.size();
    vector<QString> playerName;
    vector<CardSequence> tricks;

    for (int playerIndex=0; playerIndex < size; ++playerIndex)
    {
        tricks.push_back(m_cardsAcquired[playerIndex]);
        if (m_cardsAcquired[playerIndex].isEmpty())
            playerName.push_back(m_players[playerIndex].name() + " has no tricks.");
        else
            playerName.push_back(m_players[playerIndex].name() + " has those cards for tricks.");
    }
    KardMessageDisplay summary("Trick Points Taken", tricks, playerName, m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);

    summary.exec();
    updateTable();
}

void HeartsInterface::player1CardPlayed(Kard &kard)
{
    kardPlayed(Hearts::PLAYER_ONE_INDEX, kard);
}

void HeartsInterface::player2CardPlayed(Kard &kard)
{
    kardPlayed(Hearts::PLAYER_TWO_INDEX, kard);
}

void HeartsInterface::player3CardPlayed(Kard &kard)
{
    kardPlayed(Hearts::PLAYER_THREE_INDEX, kard);
}

void HeartsInterface::player4CardPlayed(Kard &kard)
{
    kardPlayed(Hearts::PLAYER_FOUR_INDEX, kard);
}

void HeartsInterface::kardPlayed(PlayerIndexes playerIndex, Kard &kard)
{
    if (m_players[playerIndex].isTurn())
    {
        if (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[playerIndex]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
        else
        {
            promptMessage("Warning", QString("%1 is not a legal play!").arg(kard.card().toString()));
            kard.setSelected(false);
        }
    }
    else
    {
        promptMessage("Warning", QString("%1 it is not your turn.").arg(m_players[playerIndex].name()));
        kard.setSelected(false);
    }
}

void HeartsInterface::player1CardMoved()
{
    CardSequence playersHand = player1Cards->cardSequence();

    m_players[Hearts::PLAYER_ONE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Hearts::PLAYER_ONE_INDEX].hand().addCard(playersHand[index]);
}

void HeartsInterface::player2CardMoved()
{
    CardSequence playersHand = player2Cards->cardSequence();

    m_players[Hearts::PLAYER_TWO_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Hearts::PLAYER_TWO_INDEX].hand().addCard(playersHand[index]);
}

void HeartsInterface::player3CardMoved()
{
    CardSequence playersHand = player3Cards->cardSequence();

    m_players[Hearts::PLAYER_THREE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Hearts::PLAYER_THREE_INDEX].hand().addCard(playersHand[index]);
}

void HeartsInterface::player4CardMoved()
{
    CardSequence playersHand = player4Cards->cardSequence();

    m_players[Hearts::PLAYER_FOUR_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Hearts::PLAYER_FOUR_INDEX].hand().addCard(playersHand[index]);
}

void HeartsInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (event->timerId() == m_computerTimerId)
        if (m_players[Hearts::PLAYER_ONE_INDEX].isTurn())
            computerPlay(Hearts::PLAYER_ONE_INDEX);
        else if (m_players[Hearts::PLAYER_TWO_INDEX].isTurn())
            computerPlay(Hearts::PLAYER_TWO_INDEX);
        else if (m_players[Hearts::PLAYER_FOUR_INDEX].isTurn())
            computerPlay(Hearts::PLAYER_FOUR_INDEX);
        else
            QWidget::timerEvent(event);
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void HeartsInterface::computerPlay(PlayerIndexes playerIndex)
{
    HeartsPlayer *compPlayer=static_cast<HeartsPlayer *>(&m_players[playerIndex]);

    switch (playerIndex)
    {
    case Hearts::PLAYER_ONE_INDEX:
        player1Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Hearts::PLAYER_TWO_INDEX:
        player2Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Hearts::PLAYER_THREE_INDEX:
        throw KardsGTError("HeartsInterface", "computerPlay", "Player three is not a computer player.");
        break;
    case Hearts::PLAYER_FOUR_INDEX:
        player4Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Hearts::NON_PLAYER:
        throw KardsGTError("HeartsInterface", "computerPlay", "Not a real player!");
    }
}

void HeartsInterface::gameWon()
{
    QString finalScores;

    // Create the final score messages.
    finalScores = QString("Final Scores: \n %1 had %2 \n %3 had %4 \n %5 had %6 \n %7 had %8").arg(m_players[Hearts::PLAYER_ONE_INDEX].name()).arg(m_players[Hearts::PLAYER_ONE_INDEX].score()).arg(m_players[Hearts::PLAYER_TWO_INDEX].name()).arg(m_players[Hearts::PLAYER_TWO_INDEX].score()).arg(m_players[Hearts::PLAYER_THREE_INDEX].name()).arg(m_players[Hearts::PLAYER_THREE_INDEX].score()).arg(m_players[Hearts::PLAYER_FOUR_INDEX].name()).arg(m_players[Hearts::PLAYER_FOUR_INDEX].score());
    // Find the winner
    switch (playerWhoWon())
    {
    case Hearts::PLAYER_ONE_INDEX:
        promptMessage("Game Over", QString(m_players[Hearts::PLAYER_ONE_INDEX].name() + " won the game.\n" + finalScores));
        m_pProfileDatabase->setGameStatistics("hearts", false);
        break;
    case Hearts::PLAYER_TWO_INDEX:
        promptMessage("Game Over", QString(m_players[Hearts::PLAYER_TWO_INDEX].name() + " won the game.\n" + finalScores));
        m_pProfileDatabase->setGameStatistics("hearts", false);
        break;
    case Hearts::PLAYER_THREE_INDEX:
        promptMessage("Game Over", QString(m_players[Hearts::PLAYER_THREE_INDEX].name() + " won the game.\n" + finalScores));
        m_pProfileDatabase->setGameStatistics("hearts", true);
        break;
    case Hearts::PLAYER_FOUR_INDEX:
        promptMessage("Game Over", QString(m_players[Hearts::PLAYER_FOUR_INDEX].name() + " won the game.\n" + finalScores));
        m_pProfileDatabase->setGameStatistics("hearts", false);
        break;
    default:
        throw KardsGTError("HeartsInterface", "gameWon", "No one won?");
    }
}
