/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2007-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef EUCHRERULES_H
#define EUCHRERULES_H

#include "rulebase.h"

class CardSequence;

/**
 * The rules for a game of euchre.
 *
 * @author John Schneiderman
 */
class EuchreRules : public RuleBase
{
public:
    /**
     * The public constants used by other classes.
     * @param WANT_TURN_UP means a player wants the turn-up card suit for trump.
     * @param PLAY_ALONE means a player wants to play alone.
     * @param DEFEND_ALONE mean a player wants to defend alone.
     * @param WANT_TURN_UP_CARD means the dealer wants the turn-up card for his hand.
     * @param DOES_NOT_DECIDE means the player has chosen not to do something.
     */
    enum PublicConstants
    {
        WANT_TURN_UP,
        PLAY_ALONE,
        DEFEND_ALONE,
        WANT_TURN_UP_CARD,
        DOES_NOT_DECIDE
    };

    /**
     * Default constructor.
     */
    EuchreRules();
    /**
     * Default destructor.
     */
    ~EuchreRules();
    /**
     * This determines if the conditions have been met for the game to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isGameOver(const PlayerList& players) const;
    /**
     * This determines if a card being played is a legal play.
     * @param sequence is the current play sequence.
     * @param cardPlayed is the card the player wants to player.
     * @param player is the player who played the card.
     * @return true if the card played is a legal play, false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence& sequence, const Card& cardPlayed, const Player& player) const;
    /**
     * This determines if the conditions have been met for the phase to end.
     * @param players are the players in the game.
     * @param playSequence is the current play sequence.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isPhaseOver(const PlayerList &players, const CardSequence &playSequence) const;
    /**
     * This determines if the conditions have been met for the round to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isRoundOver(const PlayerList& players) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual int cardValue(const Card& ) const;
    /**
     * Gives the rank order value for a card.
     * @param card is the card we wish to examine.
     * @return the rank value of the card.
     */
	virtual int rankValue(const Card &card) const;
    /**
     * Gives the maximum number of players allowed in the game.
     * @return the maximum number of players.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players need for the game.
     * @return the minimum number of players.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the number of cards to deal to a player.
     * @param sizeOfHand is the current size of the players hand.
     * @return the number of cards to deal.
     */
    virtual int numberOfCardsToDeal(int sizeOfHand) const;
    /**
     * Gives the score needed to win the game.
     * @return Gives the score needed to "win" the game.
     */
    virtual int winningGameScore(int numberOfPlayers = -1) const;
    /**
     * Gives the trump suit.
     * @return the trump suit.
     */
    Card::Suit trumpSuit() const;
    /**
     * Gives the left bower suit.
     * @return the left bower suit.
     */
    Card::Suit leftBowerSuit() const;
    /**
     * Sets the trump suit and left bower suit.
     * @param trumpSuit is the desired trump suit.
     */
    void setTrumpSuit(Card::Suit trumpSuit);
    /**
     * Gives the number of points needed to win the round.
     * @return the points needed to win the round.
     */
    int pointsToWinRound() const;
    /**
     * Determines if a score is a march.
     * @param score is the score to examine.
     * @return true if the score is a march, false elsewise.
     */
    bool isMarch(int score) const;
    /**
     * The number of points won if the maker is playing alone and marches.
     * @return the number of points.
     */
    int makerAloneMarchPoint() const;
    /**
     * The number of points won if the maker plays alone and doesn't march.
     * @return the number of points.
     */
    int makerAlonePoint() const;
    /**
     * The number of points won if the making-team marches.
     * @return the number of points.
     */
    int makerMarchPoint() const;
    /**
     * The number of points won if the making-team doesn't march.
     * @return the number of points.
     */
    int makerPoint() const;
    /**
     * The number of points won if the defender defends alone and marches.
     * @return the number of points.
     */
    int defenderAloneMarchPoint() const;
    /**
     * The number of points won if the defender defends alone and doesn't march.
     * @return the number of points.
     */
    int defenderAlonePoint() const;
    /**
     * The number of points won if the defender doesn't march.
     * @return the number of points.
     */
    int defenderPoint() const;
    /**
     * Determines if a card is the left bower.
     * @param card is the card to examine.
     * @return true if the card is the left bower, false elsewise.
     */
    bool isLeftBower(const Card &card) const;
    /**
     * This sets if a player is defending or playing alone.
     * @param alone is true if a player is defending or playing alone, false elsewise.
     */
    void setIsAlone(bool alone);

private:
    /**
     * These are the constants used privately in the game.
     * @param MINIMUM_PLAYERS is the minimum players needed for the game.
     * @param MAXIMUM_PLAYERS is the maximum players needed for the game.
     * @param PHASE_OVER is the number cards needed for the phase to be over.
     * @param PHASE_OVER_LONE_MAKER is the number cards needed for the phase to be over with a lone maker.
     * @param POINTS_TO_WIN_ROUND is the number of points needed to win the round.
     * @param MARCH is the points needed for a march.
     * @param MAKER_ALONE_MARCH is the points awarded for a lone maker marching.
     * @param MAKER_ALONE is the points awarded for a lone maker.
     * @param MAKER_MARCH is the points awarded for the making-team to march.
     * @param MAKER is the points awarded for the making-team.
     * @param DEFENDER_ALONE_MARCH is the points awarded for a defender marching alone.
     * @param DEFENDER_ALONE is the points awarded for a defender playing alone.
     * @param DEFENDER is the points awarded for defending.
     * @param POINTS_TO_WIN is the number of points needed to win the game.
     */
    enum PrivateConstants
    {
        MINIMUM_PLAYERS = 4,
        MAXIMUM_PLAYERS = 4,
        PHASE_OVER = 4,
        PHASE_OVER_LONE_MAKER = 3,
        POINTS_TO_WIN_ROUND = 3,
        MARCH = 5,
        MAKER_ALONE_MARCH = 4,
        MAKER_ALONE = 1,
        MAKER_MARCH = 2,
        MAKER = 1,
        DEFENDER_ALONE_MARCH = 4,
        DEFENDER_ALONE = 2,
        DEFENDER = 2,
        POINTS_TO_WIN = 5
    };
    /// @param m_trumpSuit is the trump suit.
    Card::Suit m_trumpSuit;
    /// @param m_leftBowerSuit is the left bower suit.
    Card::Suit m_leftBowerSuit;
    /// @param m_alone is true if the maker is playing alone, false elsewise.
    bool m_alone;
};
#endif
