/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "crazyeightsrules.h"
#include "playerlist.h"
#include "kardsgterror.h"
#include "cardpile.h"

#include <QtDebug>

CrazyEightsRules::CrazyEightsRules()
{}

CrazyEightsRules::~CrazyEightsRules()
{}

bool CrazyEightsRules::isGameOver(const PlayerList& players) const
{
    int winningScore=WINNING_POINTS_PER_PLAYER * players.size();

    qDebug() << "     CrazyEightsRules, Is Game Over: " << winningScore;
    for (int index=0; index < players.size(); ++index)
        if (players[index].score() >= winningScore)
            return true;
    return false;
}

bool CrazyEightsRules::isLegalPlay(const CardSequence& playSequence, const Card& cardPlayed, const Player& player) const
{
    qDebug() << "     CrazyEightsRules, Play sequence is: " << playSequence;
    qDebug() << "     CrazyEightsRules, CardPlayed is: " << cardPlayed;
    if (! player.hand().hasCard(cardPlayed))
    {
        qDebug() << "          CrazyEightsRules, Doesn't have the card.";
        return false;
    }
    if (! player.isTurn())
    {
        qDebug() << "          CrazyEightsRules, It's not your turn.";
        return false;
    }
    if (playSequence.front().suit() == cardPlayed.suit())
    {
        qDebug() << "          CrazyEightsRules, Suits are the same.";
        return true;
    }
    if (playSequence.front().rank() == cardPlayed.rank())
    {
        qDebug() << "          CrazyEightsRules, Ranks are the same";
        return true;
    }
    if (cardPlayed.rank() == Card::EIGHT)
    {
        qDebug() << "          CrazyEightsRules, An Eight was played";
        return true;
    }
    qDebug() << "     CrazyEightsRules, Invalid play.";
    return false;
}

bool CrazyEightsRules::isPhaseOver(const PlayerList &, const CardSequence &) const
{
    throw KardsGTError("CrazyEightsRules", "isPhaseOver", "This is not implemented!");
    return false;
}

bool CrazyEightsRules::isRoundOver(const PlayerList& players) const
{
    for (int index=0; index < players.size(); ++index)
        if (players[index].hand().isEmpty())
            return true;
    return false;
}

int CrazyEightsRules::cardValue(const Card& card) const
{
    int points=0;

    switch (card.rank())
    {
    case Card::EIGHT:
        points=EIGHT;
        break;
    case Card::ACE:
        points=ACE;
        break;
    case Card::KING:
    case Card::QUEEN:
    case Card::JACK:
        points=FACE_CARD;
        break;
    default:
        points=card.rank() + 1; // Starts at zero
    }
    return points;
}

int CrazyEightsRules::rankValue(const Card& card) const
{
    return card.rank();
}

int CrazyEightsRules::maximumNumberOfPlayers() const
{
    return MAX_PLAYERS;
}

int CrazyEightsRules::minimumNumberOfPlayers() const
{
    return MIN_PLAYERS;
}

int CrazyEightsRules::numberOfCardsToDeal(int numberOfPlayers) const
{
    if (numberOfPlayers == 2)
        return CARDS_FOR_TWO;
    else
        return CARDS_FOR_THREE_TO_SEVEN;
}

int CrazyEightsRules::winningGameScore(int numberOfPlayers) const
{
    return WINNING_POINTS_PER_PLAYER * numberOfPlayers;
}

bool CrazyEightsRules::isRoundOverFromBlock(const PlayerList &players, const CardPile &deck, const CardSequence& playSequence) const
{
    if (deck.isEmpty()) // Players can only pass
    {
        for (int index=0, size=players.size(); index < size; ++index)
            if (playerHasLegalPlay(players[index].hand(), playSequence, players[index]))
                return false; // A player can play, not a block case
    }
    else
        return false; // Players can draw a card if they can't play
    return true; // No one can play a card
}

bool CrazyEightsRules::playerHasLegalPlay(const CardSequence &hand, const CardSequence& playSequence, const Player &player) const
{
    Player testPlayer=player;

    testPlayer.setTurn(true);
    for (int index=0, size=hand.size(); index < size; ++index)
        if (isLegalPlay(playSequence, hand[index], testPlayer))
            return true; // Player has a legal play
    return false; // A player does not have a legal play
}
