/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CRAZY_EIGHTS_INTERFACE_H
#define CRAZY_EIGHTS_INTERFACE_H

#include "crazyeights.h"
#include "ui_crazyeightsinterface.h"
#include "userprofiledatabase.h"
#include "card.h"

#include <vector>
using std::vector;

#include <QWidget>

/**
 * This is the interface for a game of Crazy Eights.
 *
 * @author John Schneiderman
 */
class CrazyEightsInterface: public QWidget, public Ui_CrazyEightsInterface, public CrazyEights
{
    Q_OBJECT

public:
    /**
     * This is the general constructor.
     * @param profileDatabase is the user profile database.
     * @param parent is our parent widget.
     */
    CrazyEightsInterface(UserProfileDatabase &profileDatabase, QWidget *parent = 0);
    /**
     * General destructor.
     */
    ~CrazyEightsInterface();
    /**
     * Updates the interface.
     */
    void updateTable();
    /**
     * This displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * This prompts a message.
     * @param caption is caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * This requests the player to declare a suit to play with.
     * @return the suit the player selects.
     */
    Card::Suit declareSuit();
    /**
     * This displays the points each player had in their hands, and who one the round.
     * @param playerHandPoints is the points in each players hand ordered by the player list order.
     * @param winningSum is the number of points the winner won.
     */
    void displayRoundSummary(const vector<int> &playerHandPoints, int winningSum);
    /**
     * This displays who won the game, and what their score was.
     * @param playerIndex 
     */
    void gameOver(int playerIndex);
    /**
     * This displays a message that the game has enter a block state.
     */
    void roundOverFromBlock();

public slots:
    /**
     * This displays the current scores of all players.
     */
    void displayScores();

protected slots:
    /**
     * This slot handles the kard selected by a player.
     * @param kard is the card the player selected.
     */
    void playerCardPlayed(Kard &kard);
    /**
     * This slots saves the player's new card arrangement.
     */
    void playerCardMoved();
    /**
     * This slot draws a card for the player.
     * @param kard is the kard the player selected.
     */
    void drawKard(Kard &kard);
    /**
     * This slot passes a players turn.
     */
    void passDraw();

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence.
     * @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths
    {
        COMPUTER_PLAYER_TIME=1500,
        CLEARING_DELAY_TIME=500
    };
    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;
    /// @param m_pUserProfile is the user profile database.
    const UserProfileDatabase *m_pUserProfile;
    /// @param m_kardPlayers are the images of the players.
    vector<KardPlayer *> m_kardPlayers;
    /// @param m_playersCards are the hand containers for each player.
    vector<KardSequence *> m_playersCards;

    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
    /**
     * This handles the computer card playing.
     */
    void computerPlay();
};
#endif
