/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CRAZY_EIGHTS_H
#define CRAZY_EIGHTS_H

#include "gamebase.h"
#include "crazyeightsrules.h"
#include "carddeck.h"

class CrazyEightsInterface;

/**
 * This controls and keeps track of the state of the game Crazy Eights.
 *
 * @note The human player is the last player in the player index
 *
 * @author John Schneiderman
 */
class CrazyEights: public GameBase
{
public:
    /**
     * The constructor for the Crazy Eights.
     * @param gameInterface is the interface for the game.
     * @param profileDatabase is the user profile database.
     */
    CrazyEights(CrazyEightsInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * The destructor.
     */
    ~CrazyEights();
    /**
     * This starts the game of Crazy Eights.
     */
    virtual void startPlay();
    /**
     * This saves the game of Crazy Eights.
     * @param filename is the name of the file to save the game to.
     * @return true if we successfully save the game, false elsewise.
     */
    virtual bool save(const QString &filename);
    /**
     * This loads the game of Crazy Eights.
     * @param filename is the name of the file to load.
     * @return true if we load the game successfully, false elsewise.
     */
    virtual bool load(const QString &filename);
    /**
     * This gives the minimum number of players needed to play the game of Crazy Eights.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * This gives the maximum number of players allowed to play the game of Crazy Eights.
     * @return the maximum number of players.
     */
    virtual int maximumPlayers() const;

protected:
    /// @param m_rules is the rules to the game of Crazy Eights.
    CrazyEightsRules m_rules;
    /// @param m_deck is the deck of cards for Crazy Eights
    CardDeck m_deck;

    /**
     * Handles a card being played.
     * @param card is the card the player wants to play.
     */
    virtual bool cardPlayed(const Card &card);
    /**
     * Deals out the cards to each player.
     */
    virtual void deal();
    /**
     * Disables the current player, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * Determines and handles the various stages in game play.
     * @return the last major GameStatusCode value after handling the changes.
     */
    virtual int handleGameStatus();
    /**
     * This draws the card a card for the current player.
     */
    void drawCard();
    /**
     * This passes the turn for the current player.
     */
    void passPlay();

private:
    /**
     * These are the game ending states for Crazy Eights.
     * @param NO_CHANGE indicates there has been no change in the current state.
     * @param PHASE_ENDED indicates that the phase has ended.
     * @param ROUND_ENDED indicates that the round has ended.
     * @param GAME_OVER indicates that the game has ended.
     */
    enum GameStatusCodes
    {
        NO_CHANGE = 0,
        PHASE_ENDED = 1,
        ROUND_ENDED = 2,
        GAME_OVER = 3
    };
    /// @param m_pInterface is the interface to our graphical interface.
    CrazyEightsInterface *m_pInterface;

    /**
     * Handles the end of round, tabulates the winner of the round and the points he won.
     */
    void handleEndOfRound();
    /**
     * Calculates the number of points in a given hand based on the ranks of the cards.
     * @param hand is the hand to calculate.
     * @return sum of the points in the hand.
     */
    int calculateHandPoints(const CardSequence &hand) const;
    /**
     * Handles the end of the game, determines the winner, and update the game statistic database.
     */
    void handleEndOfGame();
};
#endif
