/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "cardsequence.h"
#include "kardsgterror.h"

#include <QString>
#include <QTextStream>
#include <QtDebug>

CardSequence::CardSequence(): m_sequence()
{}

CardSequence::CardSequence(const Card &card)
{
    m_sequence.clear();
    addCard(card);
}

void CardSequence::addCard(const Card &card)
{
    m_sequence.push_back(card);
}

void CardSequence::addCardToFront(const Card &card)
{
    m_sequence.push_front(card);
}

bool CardSequence::isEmpty() const
{
    Card invalidCard; // The Rank and Suit Error card.

    if (m_sequence.empty())
        return true;
    // Test to see if our sequence is filled with invalid cards
    for (int index = 0; index < static_cast<int>(m_sequence.size()); ++ index)
        if (m_sequence[index] != invalidCard)
            return false;
    return true;
}

bool CardSequence::removeCard(const Card &card)
{
    iterator pos;

    // Find our card
    pos=find(m_sequence.begin(), m_sequence.end(), card);
    // If we've found it remove it.
    if (pos != m_sequence.end())
    {
        m_sequence.erase(pos);
        return true;
    }
    else
        return false;
}

bool CardSequence::hasCard(const Card &card) const
{
    const_iterator pos;

    // Find our card
    pos=find(m_sequence.begin(), m_sequence.end(), card);
    if (pos != m_sequence.end())
        return true;
    else
        return false;
}

bool CardSequence::hasSuit(const Card::Suit &suit) const
{
    for (const_iterator pos = m_sequence.begin(); pos < m_sequence.end(); pos++)
        if (pos->suit() == suit)
            return true;
    return false;
}

bool CardSequence::hasRank(const Card::Rank &rank) const
{
    for (const_iterator pos = m_sequence.begin(); pos < m_sequence.end(); pos++)
        if (pos->rank() == rank)
            return true;
    return false;
}

int CardSequence::size() const
{
    return m_sequence.size();
}

const Card & CardSequence::operator[](int index) const
{
    if ((index < 0) || (index >= size()))
        throw KardsGTError(QString("CardSequence"), QString("const operator[]"), QString("An index of %1 is out of bounds.").arg(index));
    else
        return m_sequence[index];
}

Card& CardSequence::operator[](int index)
{
    if ((index < 0) || (index >= size()))
        throw KardsGTError(QString("CardSequence"), QString("operator[]"), QString("An index of %1 is out of bounds.").arg(index));
    else
        return m_sequence[index];
}

bool CardSequence::operator==(const CardSequence &compare) const
{
    int size=m_sequence.size();

    if (size != compare.size())
        return false;

    for (int i=0; i < size; ++i)
        if (m_sequence[i] != compare[i])
            return false;
    return true;
}

bool CardSequence::operator<(const CardSequence &compare) const
{
    if (size() < compare.size())
        return true;
    else
        return false;
}

CardSequence CardSequence::operator+(const CardSequence &rhs) const
{
    CardSequence combined;

    combined += *this;
    combined += rhs;
    return combined;
}

CardSequence & CardSequence::operator+=(const CardSequence &rhs)
{
    for (int i=0; i < rhs.size(); ++i)
        addCard(rhs[i]);
    return *this;
}

CardSequence CardSequence::operator-(const CardSequence &rhs) const
{
    CardSequence combined;

    combined = *this;
    combined -= rhs;
    return combined;
}

CardSequence & CardSequence::operator-=(const CardSequence &rhs)
{
    for (int i=0; i < rhs.size(); ++i)
        removeCard(rhs[i]);
    return *this;
}

void CardSequence::sortBySuit()
{
    int size=m_sequence.size();
    deque<CardSequence> sequenceBySuits;

    // Create a holder for each suit
    for (int i=0; i < Card::NUMBER_OF_SUITS; ++i)
        sequenceBySuits.push_back(CardSequence());
    // Place our cards into the appropriate suit sequence.
    for (int i=0; i < size; ++i)
        sequenceBySuits[m_sequence[i].suit()].addCard(m_sequence[i]);
    // Sort each of our suit holder's by rank
    for (int i=0; i < Card::NUMBER_OF_SUITS; ++i)
        sort(sequenceBySuits[i].begin(), sequenceBySuits[i].end());
    // Reset m_sequence and add the newly sorted cards.
    m_sequence.clear();
    for (int i=0; i < Card::NUMBER_OF_SUITS; ++i)
    {
        size = sequenceBySuits[i].size();
        for (int j=0; j < size; ++j)
            m_sequence.push_back(sequenceBySuits[i][j]);
    }
}

void CardSequence::clear()
{
    m_sequence.clear();
}

const Card& CardSequence::front() const
{
    return m_sequence.front();
}

Card& CardSequence::front()
{
    return m_sequence.front();
}

Card& CardSequence::back()
{
    return m_sequence.back();
}

const Card& CardSequence::back() const
{
    return m_sequence.back();
}

CardSequence::iterator CardSequence::begin()
{
    return m_sequence.begin();
}

CardSequence::const_iterator CardSequence::begin() const
{
    return m_sequence.begin();
}

CardSequence::iterator CardSequence::end()
{
    return m_sequence.end();
}

CardSequence::const_iterator CardSequence::end() const
{
    return m_sequence.end();
}

ostream& operator<<(ostream &out, const CardSequence &sequence)
{
    if (sequence.isEmpty())
        return out;

    for (int i=0; i < (sequence.size() - 1); ++i)
        out << sequence[i] << " ";
    out << sequence[sequence.size() - 1];
    return out;
}

istream& operator>>(istream &in, CardSequence &sequence)
{
    Card card;

    in >> card;
    while ((! in.fail()) && (! card.isEmpty()))
    {
        sequence.addCard(card);
        in >> card;
    }
    return in;
}

QTextStream& operator<<(QTextStream &out, const CardSequence &sequence)
{
    for (int i=0; i < sequence.size(); ++i)
        out << sequence[i] << " ";
    out << Card();
    return out;
}

QDebug& operator<<(QDebug &out, const CardSequence &sequence)
{
    for (int i=0; i < sequence.size(); ++i)
        out << sequence[i] << " ";
    out << Card();
    return out;
}

QTextStream& operator>>(QTextStream &in, CardSequence &sequence)
{
    Card card;

    sequence.clear();
    in >> card;
    while (! card.isEmpty())
    {
        sequence.addCard(card);
        in >> card;
    }
    return in;
}
