/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CARDDECK_H
#define CARDDECK_H

#include "cardpile.h"
#include "card.h"

#include <vector>
using std::vector;

/**
 * This is a deck of cards.
 *
 * @author John Schneiderman
 */
class CardDeck: public CardPile
{
public:
    /**
     * This constructor creates a full-ready deck.
     */
    CardDeck();
    /**
     * This constructor creates a full-ready deck.
     * @param ranks are the ranks to exclude from the deck.
     */
    CardDeck(vector<Card::Rank> ranks);
    /**
     * Fills any missing cards in the deck, and then shuffles the deck.
     */
    void shuffle();
    /**
     * This removes a card off the top of the deck.
     * @return the card removed from the top of the deck.
     */
    Card dealCard();
    /**
     * This randomly cuts the deck, and places the bottom half on top of the other half.
     * @note That the cut is guaranteed to be at least GUARANTEE_CUT_SIZE.
     */
    void cutDeck();
    /**
     * This removes all cards of a specified rank from the deck.
     * @param rank is the ranks to remove.
     * @note when this method is called, the deck is recreated.
     */
    void removeCards(Card::Rank rank);

protected:
    /**
     * This creates an ordered deck of cards based the on suit and rank orders listed in the @ref Card class.
     */
    void createDeck();
    /**
     * Finds all the missing cards in the deck.
     * @return the missing cards from the deck.
     */
    CardSequence findMissingCards() const;

private:
    /**
     * The constants used in the card deck.
     * @param GUARANTEE_CUT_SIZE is the minimum number of cards that will be used when cutting the deck.
     * @param DECK_SIZE is the size of the deck.
     */
    enum Constants { GUARANTEE_CUT_SIZE = 10, DECK_SIZE = Card::NUMBER_OF_SUITS * Card::NUMBER_OF_RANKS };

    /// @param m_removedRanks is the ranks removed from the deck.
    vector<Card::Rank> m_removedRanks;

    /**
     * Determines if a card is allowed in the deck
     * @return true if the card is allowed, false if it is not.
     */
    bool allowedCard(const Card &card) const;
};
#endif
