/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardplaysequence.h"
#include "kardsgterror.h"
#include "kard.h"

#include <qlayout.h>
#include <qpainter.h>
#include <qpixmap.h>

KardPlaySequence::KardPlaySequence(QWidget *parent, const char * name): QWidget(parent, name), m_kardSequences()
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_numberOfPlayers = 0;
    m_faceUp = false;
    m_cardImagePath = "";
    m_cardBackImage = "";
    createLayout();
    setAcceptDrops(true);
}

KardPlaySequence::KardPlaySequence(int numberOfPlayers, QWidget *parent, const char * name): QWidget(parent, name), m_kardSequences()
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_numberOfPlayers = numberOfPlayers;
    m_faceUp = false;
    m_cardImagePath = "";
    m_cardBackImage = "";
    createLayout();
    setAcceptDrops(true);
}

KardPlaySequence::~KardPlaySequence()
{
    clearLayout();
}

QSize KardPlaySequence::sizeHint()
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

int KardPlaySequence::numberOfPlayers() const
{
    return m_numberOfPlayers;
}

bool KardPlaySequence::isFaceUp() const
{
    return m_faceUp;
}

void KardPlaySequence::setNumberOfPlayers(int numberOfPlayers)
{
    clearLayout();
    m_numberOfPlayers = numberOfPlayers;
    createLayout();
    update();
    updateGeometry();
}

void KardPlaySequence::setFaceUp(bool faceUp)
{
    int size = m_kardSequences.size();

    m_faceUp = faceUp;
    for (int i = 0; i < size; ++i)
        m_kardSequences[i]->setFaceUp(m_faceUp);
    update();
    updateGeometry();
}

const CardSequence& KardPlaySequence::cardSequence(int playerIndex) const
{
    if ((playerIndex >= m_numberOfPlayers) || (playerIndex < 0))
        throw KardsGTError("KardPlaySequence", "cardSequence", "The playerIndex value is out of range.");

    return m_kardSequences[playerIndex]->cardSequence();
}

void KardPlaySequence::setCardSequence(const CardSequence &sequence, int starterPlayerIndex)
{
    if ((starterPlayerIndex >= m_numberOfPlayers) || (starterPlayerIndex < 0))
        throw KardsGTError("KardPlaySequence", "setCardSequence", "The starterPlayerIndex value is out of range.");
    int size = sequence.size();
    vector<CardSequence> playSequences;

    // Create a play sequence for each player
    for (int i=0; i < m_numberOfPlayers; ++i)
        playSequences.push_back(CardSequence());
    // Add each card that the player played to their play sequence.
    for (int cardIndex=0, playerIndex = starterPlayerIndex; cardIndex < size; playerIndex = (playerIndex + 1) % m_numberOfPlayers, ++cardIndex)
        playSequences[playerIndex].addCard(sequence[cardIndex]);
    // Set their new play seqeuence.
    for (int index = 0; index < m_numberOfPlayers; ++index)
        m_kardSequences[index]->setCardSequence(playSequences[index]);
}

void KardPlaySequence::setCardSequence(const CardSequence &sequence, vector<int> playerCardIndexes)
{
    if (sequence.size() != static_cast<int>(playerCardIndexes.size()))
        throw KardsGTError("KardPlaySequence", "setCardSequence", "The sequence and playerCardIndexes are not the same size!");
    int size = sequence.size();
    vector<CardSequence> playSequences;

    // Create a play sequence for each player
    for (int i=0; i < m_numberOfPlayers; ++i)
        playSequences.push_back(CardSequence());
    // Add each card that the player played to their play sequence.
    for (int cardIndex=0; cardIndex < size; ++cardIndex)
        playSequences[playerCardIndexes[cardIndex]].addCard(sequence[cardIndex]);
    // Set their new play seqeuence.
    for (int index = 0; index < m_numberOfPlayers; ++index)
        m_kardSequences[index]->setCardSequence(playSequences[index]);
}

void KardPlaySequence::clear()
{
    int size = m_kardSequences.size();

    for (int i = 0; i < size; ++i)
        m_kardSequences[i]->clear();
    update();
    updateGeometry();
}

void KardPlaySequence::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());
    draw(painter);
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void KardPlaySequence::createLayout()
{
    m_pLayout=new QVBoxLayout(this, MARGIN, SPACING, "KardPlaySequenceLayout");
    for (int i=0; i < m_numberOfPlayers; ++i)
    {
        m_kardSequences.push_back(new KardSequence(this, QString("KardPlaySequence widget #%1").arg(i)));
        m_kardSequences[i]->setMinimumSize(Kard::PREFERRED_WIDTH, Kard::PREFERRED_HEIGHT / 4);
        m_kardSequences[i]->setMaximumHeight(Kard::PREFERRED_HEIGHT);
        m_kardSequences[i]->setSortPopup(false);
        m_kardSequences[i]->setBackImage(m_cardBackImage);
        m_kardSequences[i]->setCardImagePath(m_cardImagePath);
    }
    QHBoxLayout *player1Layout=new QHBoxLayout(0, MARGIN, SPACING, "Player layout: player 1");
    QHBoxLayout *player2Layout=new QHBoxLayout(0, MARGIN, SPACING, "Player layout: player 2");
    QHBoxLayout *player2And4Layout;
    QHBoxLayout *player3Layout;

    switch (m_numberOfPlayers)
    {
    case 0:
        break;
    case 1:
    case 2:
        player1Layout->addWidget(m_kardSequences[0]);
        player2Layout->addWidget(m_kardSequences[1]);
        m_pLayout->addLayout(player1Layout);
        m_pLayout->addLayout(player2Layout);
        break;
    case 3:
    case 4:
        player2And4Layout=new QHBoxLayout(0, MARGIN, SPACING, "Player layout: player 2 & 4");
        player3Layout=new QHBoxLayout(0, MARGIN, SPACING, "Player layout: player 3");

        player1Layout->addWidget(m_kardSequences[0]);
        player2And4Layout->addWidget(m_kardSequences[3]);
        player2And4Layout->addWidget(m_kardSequences[1]);
        player3Layout->addWidget(m_kardSequences[2]);
        m_pLayout->addLayout(player1Layout);
        m_pLayout->addLayout(player2And4Layout);
        m_pLayout->addLayout(player3Layout);
        break;
    default:
        throw KardsGTError("KardPlaySequence", "createLayout", "Layout is not specified for that many players!");
    }
}

void KardPlaySequence::clearLayout()
{
    int size = m_kardSequences.size();

    for (int i = 0; i < size; ++i)
        delete m_kardSequences[i];
    m_kardSequences.clear();
    delete m_pLayout;
}

void KardPlaySequence::draw(QPainter &painter)
{
    int size = m_kardSequences.size();

    painter.drawRect(painter.window());
    for (int i=0; i < size; ++i)
        m_kardSequences[i]->show();
}

void KardPlaySequence::setBackImage(const QString &backImage)
{
    m_cardBackImage = backImage;
    for (int i=0; i < m_numberOfPlayers; ++i)
        m_kardSequences[i]->setBackImage(m_cardBackImage);
}

void KardPlaySequence::setCardImagePath(const QString &imagePath)
{
    m_cardImagePath = imagePath;
    for (int i=0; i < m_numberOfPlayers; ++i)
        m_kardSequences[i]->setCardImagePath(m_cardImagePath);
}
