/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef EUCHREINTERFACE_H
#define EUCHREINTERFACE_H

#include "euchre.h"
#include "euchreinterfacebase.h"

#include <qwidget.h>

class Kard;

/**
 * This is the interface for a game of euchre.
 *
 * @author John Schneiderman
 */
class EuchreInterface: public EuchreInterfaceBase, public Euchre
{
    Q_OBJECT

public:
    /**
     * This is the general constructor.
     * @param profileDatabase is the user profile database.
     * @param parent is our parent widget.
     * @param name is the name of us.
     */
    EuchreInterface(UserProfileDatabase &profileDatabase, QWidget *parent = 0, const char *name = 0);
    /**
     * General destructor.
     */
    ~EuchreInterface();
    /**
     * This updates all of the interface widgets.
     */
    void updateTable();
    /**
     * Asks the player if he wishes to have the turn-up card.
     * @return
     */
    bool makeTurnUp();
    /**
     * Asks the player to declare a suit.
     * @param suit is the suit of the turn-up card.
     * @return the suit selected, if passed SUIT_ERR.
     */
    Card::Suit selectTrump(Card::Suit suit);
    /**
     * Asks the player if he wishes to play alone.
     * @return true if he wishes to play alone, false elsewise.
     */
    bool playAlone();
    /**
     * Asks the player if he wishes to defend alone.
     * @return true if he wishes to defend alone, false elsewise.
     */
    bool defendAlone();
    /**
     * Displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     * @throw KardsGTError if the player is not found.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * Prompts a message.
     * @param caption is the caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * Shows the game winning information.
     */
    void gameWon();
    /**
     * Asks the player if he wants the turn-up card.
     * @return true if the dealer wants the turn-up card, false elsewise.
     */
    bool dealerWantsTurnUp();
    /**
     * Asks the player to discard a card from his hand.
     * @return the card selected from his hand.
     */
    Card discard();

public slots:
    /**
     * Has the player who is the maker to say so.
     */
    virtual void displayMaker();
    /**
     * Has each player say the tricks they've won.
     */
    virtual void displayTricks();
    /**
     * Displays the current team scores.
     */
    virtual void displayScores();

protected slots:
    /**
     * Handles the kard selected by player 1.
     * @param kard is the card the player selected.
     */
    void player1CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 2.
     * @param kard is the card the player selected.
     */
    void player2CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 3.
     * @param kard is the card the player selected.
     */
    void player3CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 4.
     * @param kard is the card the player selected.
     */
    void player4CardPlayed(Kard &kard);
    /**
     * Saves the player 1's new card arrangement.
     */
    void player1CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player2CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player3CardMoved();
    /**
     * Saves the player 1's new card arrangement.
     */
    void player4CardMoved();

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence.
     * @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths { COMPUTER_PLAYER_TIME=1500, CLEARING_DELAY_TIME=500 };

    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;
    /// @param m_pUserProfile is the user profile database.
    const UserProfileDatabase *m_pUserProfile;

    /**
     * Hangles the actual card played.
     * @param playerIndex is the index for the player who played a card.
     * @param kard is the card the player has selected.
     */
    void kardPlayed(PlayerIndexes playerIndex, Kard &kard);
    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
    /**
     * This handles the computer card playing.
     * @param playerIndex is the index of the player who should play a card.
     * @throw KardsGTError if the playerIndex is the third player or a non-player.
     */
    void computerPlay(PlayerIndexes playerIndex);
};
#endif
