 /***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "abigailheartsai.h"
#include "cardproperties.h"
#include "kardsgterror.h"
#include "cardsequence.h"

#include <vector>
using std::vector;

AbigailHeartsAI::AbigailHeartsAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_hand = hand;
    m_pRules = &rules;
}

AbigailHeartsAI::~AbigailHeartsAI()
{}

Card AbigailHeartsAI::selectCard() const
{
    Card card;
    bool selected = false;

    // Lead with the lowest card
    if (m_playSequence.isEmpty())
    {
        if (m_hand.hasCard(Card(Card::TWO, Card::CLUBS)))
            card = Card(Card::TWO, Card::CLUBS);
        else
            card = basicStrategies.lowestCard(m_hand);
        selected = true;
    }

    // Play lowest card of lead suit
    if (! selected)
    {
        card = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
        if (! card.isEmpty())
            selected = true;
    }

    // Play heart if can't play lead suit
    if (! selected)
    {
        card = basicStrategies.highestCardOfSuit(m_hand, Card::HEARTS);
        if (! card.isEmpty())
            selected = true;
    }

    // If can't play heart (no hearts) play lowest card
    if (! selected)
        card = basicStrategies.lowestCard(m_hand);

    return card;
}

CardSequence AbigailHeartsAI::passCards() const
{
    CardProperties cardProp(m_hand);
    CardSequence testSuit;
    CardSequence cards; // These are the cards to pass
    Card testCard;
    bool selected = false;

    // Pass all high spades (A, K), unless guarded (3 or more spades)
    testSuit = cardProp.suits(Card::SPADES);
    if (! testSuit.isEmpty() && (testSuit.size() < 3))
    {
        if (m_hand.hasCard(Card(Card::ACE, Card::SPADES)))
            cards.addCard(Card(Card::ACE, Card::SPADES));
        if (m_hand.hasCard(Card(Card::KING, Card::SPADES)))
            cards.addCard(Card(Card::KING, Card::SPADES));
        if (cards.size() == 3)
            selected = true;
    }

    // Pass all high hearts, unless they're guarded
    testSuit = cardProp.suits(Card::HEARTS);
    if ((! selected) && (! testSuit.isEmpty() && (testSuit.size() < 3)))
    {
        if (m_hand.hasCard(Card(Card::ACE, Card::HEARTS)))
            cards.addCard(Card(Card::ACE, Card::HEARTS));
        if (m_hand.hasCard(Card(Card::KING, Card::HEARTS)))
            cards.addCard(Card(Card::KING, Card::HEARTS));
        if (cards.size() == 3)
            selected = true;
    }

    // Select remaining high cards
    if (! selected)
        while (cards.size() < 3)
        {
            testCard = basicStrategies.randomCardsWithNoLegalChecks(1, m_hand).front();
            if (! cards.hasCard(testCard))
                cards.addCard(testCard);
        }
    return cards;
}
