/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "euchreinterface.h"
#include "kardsequence.h"
#include "kardplayer.h"
#include "kardplaysequence.h"
#include "kardpile.h"
#include "userprofiledatabase.h"
#include "suitrequest.h"
#include "kardsgterror.h"
#include "kard.h"
#include "euchreplayer.h"
#include "kardselection.h"

#include <qmessagebox.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qimage.h>

EuchreInterface::EuchreInterface(UserProfileDatabase &profileDatabase, QWidget *parent, const char *name): EuchreInterfaceBase(parent, name), Euchre(this, profileDatabase)
{
    m_pUserProfile = &profileDatabase;
    setBackgroundColor(darkGreen);
    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
    connect(player3Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player3CardPlayed(Kard &)));
    connect(player4Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player4CardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved()), this, SLOT(player1CardMoved()));
    connect(player2Cards, SIGNAL(kardMoved()), this, SLOT(player2CardMoved()));
    connect(player3Cards, SIGNAL(kardMoved()), this, SLOT(player3CardMoved()));
    connect(player4Cards, SIGNAL(kardMoved()), this, SLOT(player4CardMoved()));

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
    discardPile->setFaceUp(true);
}

EuchreInterface::~EuchreInterface()
{}

void EuchreInterface::updateTable()
{
    // Set the card back images
    player1Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player2Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player3Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player4Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    discardPile->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    deck->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));

    // Set the card front path
    player1Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player2Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player3Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player4Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    discardPile->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    deck->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()));

    // Update player hands
    player1Cards->setCardSequence(m_players[Euchre::PLAYER_ONE_INDEX].hand());
    player2Cards->setCardSequence(m_players[Euchre::PLAYER_TWO_INDEX].hand());
    player3Cards->setCardSequence(m_players[Euchre::PLAYER_THREE_INDEX].hand());
    player4Cards->setCardSequence(m_players[Euchre::PLAYER_FOUR_INDEX].hand());
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        discardPile->setCardSequence(m_playSequence, m_playOrder);
    deck->setCardPile(m_deck);
    deck->setTopCardUp(m_turnUpAccepted);
    switch (m_trumpSuit)
    {
    case Card::SPADES:
        trump->setPixmap(QImage::fromMimeSource("spade.png"));
        break;
    case Card::HEARTS:
        trump->setPixmap(QImage::fromMimeSource("heart.png"));
        break;
    case Card::CLUBS:
        trump->setPixmap(QImage::fromMimeSource("club.png"));
        break;
    case Card::DIAMONDS:
        trump->setPixmap(QImage::fromMimeSource("diamond.png"));
        break;
    case Card::SUIT_ERR:
        trump->setPixmap(QImage::fromMimeSource("back00.png"));
        break;
    default:
        throw KardsGTError("EuchreInterface", "updateTable", "Failed to determine trump suit!");
    }

    // Update table info
    // Player 1
    player1->setPlayerImage(m_players[Euchre::PLAYER_ONE_INDEX].name().lower() + ".png");
    player1->setDealer(m_players[Euchre::PLAYER_ONE_INDEX].isDealer());
    player1->setTurn(m_players[Euchre::PLAYER_ONE_INDEX].isTurn());
    player1->setLevel(m_players[Euchre::PLAYER_ONE_INDEX].level());
    // player 2
    player2->setPlayerImage(m_players[Euchre::PLAYER_TWO_INDEX].name().lower() + ".png");
    player2->setDealer(m_players[Euchre::PLAYER_TWO_INDEX].isDealer());
    player2->setTurn(m_players[Euchre::PLAYER_TWO_INDEX].isTurn());
    player2->setLevel(m_players[Euchre::PLAYER_TWO_INDEX].level());
    // player 3
    player3->setPlayerImage(m_pUserProfile->userImage(m_players[Euchre::PLAYER_THREE_INDEX].name()));
    player3->setDealer(m_players[Euchre::PLAYER_THREE_INDEX].isDealer());
    player3->setTurn(m_players[Euchre::PLAYER_THREE_INDEX].isTurn());
    player3->setLevel(m_players[Euchre::PLAYER_THREE_INDEX].level());
    // player 4
    player4->setPlayerImage(m_players[Euchre::PLAYER_FOUR_INDEX].name().lower() + ".png");
    player4->setDealer(m_players[Euchre::PLAYER_FOUR_INDEX].isDealer());
    player4->setTurn(m_players[Euchre::PLAYER_FOUR_INDEX].isTurn());
    player4->setLevel(m_players[Euchre::PLAYER_FOUR_INDEX].level());
}

bool EuchreInterface::makeTurnUp()
{
    QString acceptMessage;

    if (m_players[PLAYER_ONE_INDEX].isDealer())
        acceptMessage = "assist";
    else if (m_players[PLAYER_THREE_INDEX].isDealer())
        acceptMessage = "take it up";
    else
        acceptMessage = "order it up";
    int responce = QMessageBox::question(this, "Turn-Up Trump", QString("Would you like to %1?").arg(acceptMessage), QMessageBox::Yes, QMessageBox::No, QMessageBox::NoButton);

    if (responce == QMessageBox::Yes)
        return true;
    else
        return false;
}

Card::Suit EuchreInterface::selectTrump(Card::Suit suit)
{
    SuitRequest suitRequest(QString("Name Trump"), true, false, this, "name trump dialogue");
    Card::Suit trump;

    if (suitRequest.exec())
    {
        trump = suitRequest.suit();
        if (trump == suit)
        {
            QMessageBox::warning(this, "Trump Warning", "You cannot name trump suit to be that of the turn-up card!", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
            return selectTrump(suit);
        }
        else
            return trump;
    }
    else
        return selectTrump(suit);
}

bool EuchreInterface::playAlone()
{
    int responce = QMessageBox::question(this, "Maker Options", QString("Would you like to play alone?"), QMessageBox::Yes, QMessageBox::No, QMessageBox::NoButton);

    if (responce == QMessageBox::Yes)
        return true;
    else
        return false;
}

bool EuchreInterface::defendAlone()
{
    int responce = QMessageBox::question(this, "Defend Options", QString("Would you like to defend alone?"), QMessageBox::Yes, QMessageBox::No, QMessageBox::NoButton);

    if (responce == QMessageBox::Yes)
        return true;
    else
        return false;
}

void EuchreInterface::displayMessage(const Player &player, const QString &message)
{
    if (player.name() == m_players[Euchre::PLAYER_ONE_INDEX].name())
        player1->setCaption(message);
    else if (player.name() == m_players[Euchre::PLAYER_TWO_INDEX].name())
        player2->setCaption(message);
    else if (player.name() == m_players[Euchre::PLAYER_THREE_INDEX].name())
        player3->setCaption(message);
    else if (player.name() == m_players[Euchre::PLAYER_FOUR_INDEX].name())
        player4->setCaption(message);
    else
        throw KardsGTError("EuchreInterface", "displayMessage", "Player not found!");
}

void EuchreInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
}

void EuchreInterface::gameWon()
{
    QString finalScores;

    // Create the final score messages.
    finalScores = QString("These are the final scores:\n %1-%2 Team: %3 points\n %4-%5 Team: %6 points").arg(m_players[Euchre::PLAYER_ONE_INDEX].name()).arg(m_players[Euchre::PLAYER_THREE_INDEX].name()).arg(m_players[Euchre::PLAYER_ONE_INDEX].score()).arg(m_players[Euchre::PLAYER_TWO_INDEX].name()).arg(m_players[Euchre::PLAYER_FOUR_INDEX].name()).arg(m_players[Euchre::PLAYER_TWO_INDEX].score());
    // Find the winner
    if (m_players[Euchre::PLAYER_ONE_INDEX].score() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString("Team " + m_players[Euchre::PLAYER_ONE_INDEX].name() + "-" + m_players[Euchre::PLAYER_THREE_INDEX].name() + " won the game.\n\n" + finalScores));
        m_pProfileDatabase->setUserGamesWon(m_players[Euchre::PLAYER_THREE_INDEX].name(), "euchre", true);
    }
    else if (m_players[Euchre::PLAYER_TWO_INDEX].score() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString("Team " + m_players[Euchre::PLAYER_TWO_INDEX].name() + "-" + m_players[Euchre::PLAYER_FOUR_INDEX].name() + " won the game.\n\n" + finalScores));
        m_pProfileDatabase->setUserGamesWon(m_players[Euchre::PLAYER_THREE_INDEX].name(), "euchre", false);
    }
    else
        throw KardsGTError("EuchreInterface", "gameWon", "No one won?");
}

bool EuchreInterface::dealerWantsTurnUp()
{
    int responce = QMessageBox::question(this, "Dealer Options", QString("Would you like the turn-up card?"), QMessageBox::Yes, QMessageBox::No, QMessageBox::NoButton);

    if (responce == QMessageBox::Yes)
        return true;
    else
        return false;
}

Card EuchreInterface::discard()
{
    KardSelection *selectionInterface=new KardSelection("Select card to discard...", 1, m_players[Euchre::PLAYER_THREE_INDEX].hand(), m_pUserProfile->pathToCardImages(m_players[Euchre::PLAYER_THREE_INDEX].name()), m_pUserProfile->pathToCardBackImage(m_players[Euchre::PLAYER_THREE_INDEX].name()), this, "discardSelection");

    if (selectionInterface->exec())
        return selectionInterface->selection().front();
    else
        return discard();
}

void EuchreInterface::displayMaker()
{
    switch (m_makerIndex)
    {
    case Euchre::PLAYER_ONE_INDEX:
        player1->setCaption("I'm the maker.");
        break;
    case Euchre::PLAYER_TWO_INDEX:
        player2->setCaption("I'm the maker.");
        break;
    case Euchre::PLAYER_THREE_INDEX:
        player3->setCaption("I'm the maker.");
        break;
    case Euchre::PLAYER_FOUR_INDEX:
        player4->setCaption("I'm the maker.");
        break;
    default:
        promptMessage("Maker", "No one is the maker.");
        break;
    }
}

void EuchreInterface::displayTricks()
{
    player1->setCaption(QString("I have %1 tricks.").arg(m_tricksWon[PLAYER_ONE_INDEX]));
    player2->setCaption(QString("I have %1 tricks.").arg(m_tricksWon[PLAYER_TWO_INDEX]));
    player3->setCaption(QString("I have %1 tricks.").arg(m_tricksWon[PLAYER_THREE_INDEX]));
    player4->setCaption(QString("I have %1 tricks.").arg(m_tricksWon[PLAYER_FOUR_INDEX]));
}

void EuchreInterface::displayScores()
{
    promptMessage("Scores", QString("These are the current scores:\n %1-%2 Team: %3 points\n %4-%5 Team: %6 points").arg(m_players[Euchre::PLAYER_ONE_INDEX].name()).arg(m_players[Euchre::PLAYER_THREE_INDEX].name()).arg(m_players[Euchre::PLAYER_ONE_INDEX].score()).arg(m_players[Euchre::PLAYER_TWO_INDEX].name()).arg(m_players[Euchre::PLAYER_FOUR_INDEX].name()).arg(m_players[Euchre::PLAYER_TWO_INDEX].score()));
}

void EuchreInterface::player1CardPlayed(Kard &kard)
{
    kardPlayed(Euchre::PLAYER_ONE_INDEX, kard);
}

void EuchreInterface::player2CardPlayed(Kard &kard)
{
    kardPlayed(Euchre::PLAYER_TWO_INDEX, kard);
}

void EuchreInterface::player3CardPlayed(Kard &kard)
{
    kardPlayed(Euchre::PLAYER_THREE_INDEX, kard);
}

void EuchreInterface::player4CardPlayed(Kard &kard)
{
    kardPlayed(Euchre::PLAYER_FOUR_INDEX, kard);
}

void EuchreInterface::player1CardMoved()
{
    CardSequence playersHand = player1Cards->cardSequence();

    m_players[Euchre::PLAYER_ONE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Euchre::PLAYER_ONE_INDEX].hand().addCard(playersHand[index]);
}

void EuchreInterface::player2CardMoved()
{
    CardSequence playersHand = player2Cards->cardSequence();

    m_players[Euchre::PLAYER_TWO_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Euchre::PLAYER_TWO_INDEX].hand().addCard(playersHand[index]);
}

void EuchreInterface::player3CardMoved()
{
    CardSequence playersHand = player3Cards->cardSequence();

    m_players[Euchre::PLAYER_THREE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Euchre::PLAYER_THREE_INDEX].hand().addCard(playersHand[index]);
}

void EuchreInterface::player4CardMoved()
{
    CardSequence playersHand = player4Cards->cardSequence();

    m_players[Euchre::PLAYER_FOUR_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Euchre::PLAYER_FOUR_INDEX].hand().addCard(playersHand[index]);
}

void EuchreInterface::kardPlayed(PlayerIndexes playerIndex, Kard &kard)
{
    if (m_players[playerIndex].isTurn())
    {
        if (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[playerIndex]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
        else
        {
            promptMessage("Warning", QString("%1 is not a legal play!").arg(kard.card().toString()));
            kard.setSelected(false);
        }
    }
    else
    {
        promptMessage("Warning", QString("%1 it is not your turn.").arg(m_players[playerIndex].name()));
        kard.setSelected(false);
    }
}

void EuchreInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (event->timerId() == m_computerTimerId)
        if (m_players[Euchre::PLAYER_ONE_INDEX].isTurn())
            computerPlay(Euchre::PLAYER_ONE_INDEX);
        else if (m_players[Euchre::PLAYER_TWO_INDEX].isTurn())
            computerPlay(Euchre::PLAYER_TWO_INDEX);
        else if (m_players[Euchre::PLAYER_FOUR_INDEX].isTurn())
            computerPlay(Euchre::PLAYER_FOUR_INDEX);
        else
            QWidget::timerEvent(event);
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void EuchreInterface::computerPlay(PlayerIndexes playerIndex)
{
    EuchrePlayer *compPlayer=static_cast<EuchrePlayer *>(&m_players[playerIndex]);

    switch (playerIndex)
    {
    case Euchre::PLAYER_ONE_INDEX:
        player1Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Euchre::PLAYER_TWO_INDEX:
        player2Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Euchre::PLAYER_THREE_INDEX:
        throw KardsGTError("EuchreInterface", "computerPlay", "Player three is not a computer player.");
        break;
    case Euchre::PLAYER_FOUR_INDEX:
        player4Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Euchre::NON_PLAYER:
        throw KardsGTError("EuchreInterface", "computerPlay", "Not a real player!");
    }
}

