/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cribbage.h"
#include "cribbageinterface.h"
#include "cardsequence.h"
#include "cribbageplayer.h"
#include "kardsgterror.h"
#include "cardproperties.h"
#include "kardpile.h"
#include "kardplaysequence.h"

#include <algorithm>
#include <qstring.h>
#include <qfile.h>

Cribbage::Cribbage(CribbageInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_deck(), m_playOrder(), m_crib(), m_starter(), m_goPlayer(), m_savedHands(), m_cardSummary(), m_messageSummary()
{
    m_scoringCrib=m_inGo=false;
    m_pInterface=gameInterface;
    m_roundWinner = -1;
}

Cribbage::~Cribbage()
{}

void Cribbage::startPlay()
{
    time_t seconds;
    int rnd;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("Cribbage", "startPlay", "Wrong number of players were added to the game.");
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    rnd=(rand() % 100) + 1;
    if (rnd > 50)
        m_dealerIndex=1;
    else
        m_dealerIndex=0;
    // Deal cards
    deal();
}

bool Cribbage::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(IO_WriteOnly | IO_Translate))
        return false;
    out.setEncoding(QTextStream::UnicodeUTF8);
    out << "Game: " << "cribbage" << endl;
    // Begin GameBase Save
    out << "Players: " << endl << m_players << endl;
    out << "RoundPoints: " << m_roundPoints << endl;
    out << "DealerIndex: " << m_dealerIndex << endl;
    out << "PlaySequence: " << m_playSequence << endl;
    // End GameBase Save
    out << "PreviousScore: " << m_pInterface->previousScores().first << endl;
    out << "PreviousScore: " << m_pInterface->previousScores().second << endl;
    out << "Deck: " << m_deck << endl;
    out << "Starter: " << m_starter << endl;
    out << "Crib: " << m_crib << endl;
    out << "NumberOfSavedhands: " << m_savedHands.size() << endl;
    for (int i=0; i < static_cast<int>(m_savedHands.size()); ++i)
        out << "SavedHand:" << i << " " << m_savedHands[i] << endl;
    out << "InGo: " << m_inGo << endl;
    out << "GoPlayer: " << endl << m_goPlayer << endl;
    out << "PlayOrderSize: " << m_playOrder.size() << endl;
    out << "PlayOrder: ";
    for (int i=0; i < static_cast<int>(m_playOrder.size()); ++i)
        out << m_playOrder[i] << " ";
    out << endl;
    if (file.status() != IO_Ok)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool Cribbage::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    int numberOfSavedHands=0, intToBool=0, redPreviousScore=0, bluePreviousScore=0, value=-1, size;
    CardSequence savedHands;
    QString name, label;

    if (! file.open(IO_ReadOnly | IO_Translate))
    {
        file.close();
        return false;
    }
    in.setEncoding(QTextStream::UnicodeUTF8);
    in >> label >> name;
    if (name != "cribbage")
    {
        file.close();
        return false;
    }
    m_pInterface->resetGame();
    // Begin GameBase Load
    in >> label >> m_players;
    in >> label >> m_roundPoints;
    in >> label >> m_dealerIndex;
    in >> label >> m_playSequence;
    // End GameBase Load
    in >> label >> redPreviousScore;
    in >> label >> bluePreviousScore;
    in >> label >> m_deck;
    in >> label >> m_starter;
    in >> label >> m_crib;
    in >> label >> numberOfSavedHands;
    m_savedHands.clear();
    for (int i=0; i < numberOfSavedHands; ++i)
    {
        in >> label  >> savedHands;
        m_savedHands.push_back(savedHands);
    }
    in >> label >> intToBool;
    m_inGo = static_cast<bool>(intToBool);
    in >> label >> m_goPlayer;
    m_playOrder.clear();
    in >> label >> size;
    in >> label;
    for (int i=0; i < size; ++i)
    {
        in >> value;
        m_playOrder.push_back(value);
    }
    m_pInterface->showStarterCard(true);
    m_pInterface->updateTable();
    m_pInterface->updateStakes(-1);
    m_pInterface->setPreviousScores(redPreviousScore, bluePreviousScore);
    file.close();
    return true;
}

int Cribbage::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int Cribbage::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

void Cribbage::deal()
{
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().clear();
    nextDealer();
    m_pInterface->showStarterCard(false);
    m_pInterface->updateTable();
    m_deck.shuffle();
    for (int i=0; i < m_rules.numberOfCardsToDeal(); ++i)
        for (int j=0; j < m_players.size(); j++)
            m_players.nextPlayer().hand().addCard(m_deck.dealCard());

    //Pass to the crib
    for (int i=0; i < m_players.size(); ++i)
        discardToCrib(m_players[i]);

    //Create starter card
    m_deck.cutDeck();
    m_starter=m_deck.topCard();

    //Check for "His Heels"
    if (m_starter.rank() == Card::JACK)
    {
        m_players[m_dealerIndex].addPoint(CribbageRules::HEELS);
        m_pInterface->displayMessage(m_players[m_dealerIndex], "2 for his heels.");
    }
    saveHands();
    handleGameStatus();
    m_pInterface->showStarterCard(true);
    m_pInterface->updateTable();
}

void Cribbage::discardToCrib(Player &player)
{
    CardSequence cards;
    CribbagePlayer *cribbagePlayer=dynamic_cast<CribbagePlayer *>(&player);

    if (cribbagePlayer == 0)
    {
        player.setTurn(true);
        cards=m_pInterface->requestCards((player.isDealer() ? "Discard to your crib" : QString("Discard to %1's crib").arg(m_players[0].name())), player, CribbageRules::NUMBER_CARDS_TO_CRIB);
        for (int i=0; i < cards.size(); ++i)
        {
            player.hand().removeCard(cards[i]);
            m_crib.addCard(cards[i]);
        }
        player.setTurn(false);
    }
    else
    {
        cards=cribbagePlayer->passToCrib();
        for (int i=0; i < cards.size(); ++i)
            m_crib.addCard(cards[i]);
    }
}

void Cribbage::saveHands()
{
    m_savedHands.clear();
    m_savedHands.push_back(m_players[0].hand());
    m_savedHands.push_back(m_players[1].hand());
}

void Cribbage::restoreHands()
{
    m_players[0].hand()=m_savedHands[0];
    m_players[1].hand()=m_savedHands[1];
}

void Cribbage::cardPlayed(const Card& card)
{
    m_roundPoints+=m_rules.cardValue(card);
    m_playSequence.addCard(card);
    if (m_players.currentPlayer() == m_players[0])
        m_playOrder.push_back(0);
    else
        m_playOrder.push_back(1);
    scoreInPlay(m_players.currentPlayer());
    if (m_players.currentPlayer().hand().removeCard(card))
    {
        m_pInterface->updateTable();
        handleGameStatus();
    }
    else
        throw KardsGTError(QString("Cribbage"), QString("cardPlayed"), card.toString() + " was not found in the players hand.");
}

int Cribbage::handleGameStatus()
{
    if (m_rules.isRoundOver(m_players))
    {
        // The last card has been played
        m_pInterface->displayMessage(m_players.currentPlayer(), "2 for last card.");
        m_players.currentPlayer().addPoint(CribbageRules::LAST);
        m_playSequence.clear();
        m_playOrder.clear();
        roundOver();
        m_pInterface->updateTable();
        return 0;
    }
    if (! m_inGo)
        setupNextPlayer();
    checkGo();
    return 0;
}

void Cribbage::setupNextPlayer()
{
    m_players.currentPlayer().setTurn(false);
    m_players.nextNonEmptyPlayer();
    m_players.currentPlayer().setTurn(true);
}


void Cribbage::checkGo()
{
    // We have a forced GO situation in the players hand.
    if (! m_inGo && haveGO(m_players.currentPlayer()))
    {
        m_pInterface->displayMessage(m_players.currentPlayer(), "Go.");
        m_goPlayer=m_players.currentPlayer();
        m_inGo=true;
        setupNextPlayer();
        checkGo();
    }
    //The other player has played as many cards as he can.
    if (m_inGo && haveGO(m_players.currentPlayer()))
    {
        if (m_roundPoints < CribbageRules::MAXIMUM_PHASE_POINTS)
        {
            m_pInterface->displayMessage(m_players.currentPlayer(), QString("%1 for GO.").arg(CribbageRules::GO));
            m_players.currentPlayer().addPoint(CribbageRules::GO);
        }
        m_inGo=false;
        m_playSequence.clear();
        m_playOrder.clear();
        m_roundPoints=0;
        setupNextPlayer();
        m_pInterface->updateTable();
    }
}

bool Cribbage::haveGO(const Player &player)
{
    if (m_roundPoints == CribbageRules::MAXIMUM_PHASE_POINTS)
        return true;
    for (int i=0; i < player.hand().size(); ++i)
        if ((m_roundPoints + m_rules.cardValue(player.hand()[i])) <= CribbageRules::MAXIMUM_PHASE_POINTS)
            return false;
    return true;
}

void Cribbage::scoreInPlay(Player &player)
{
    CardSequence testSequence;
    CardProperties cardProperties(m_playSequence);
    vector<CardSequence> sequencesFound;
    int points=0, size=0;

    //Check the counts.
    if (m_roundPoints == 15)
    {
        m_pInterface->displayMessage(player, QString("%1 for 15.").arg(CribbageRules::FIFTEEN));
        points+=CribbageRules::FIFTEEN;
    }
    else if (m_roundPoints == CribbageRules::MAXIMUM_PHASE_POINTS)
    {
        m_pInterface->displayMessage(player, QString("%1 for 31.").arg(CribbageRules::THIRTY_ONE));
        points+=CribbageRules::THIRTY_ONE;
        m_playSequence.clear();
        m_playOrder.clear();
        m_roundPoints=0;
    }

    //Are there a pairs
    testSequence.clear();
    sequencesFound=cardProperties.pairs();
    m_rules.createPairTypes(sequencesFound);
    if ((!sequencesFound.empty()) && (!sequencesFound.front().isEmpty()))
    {
        for (int seqFndN=0; seqFndN < static_cast<int>(sequencesFound.size()); ++seqFndN)
        {
            // Add as many cards to our testSequence as we have in the sequence found.
            testSequence.clear();
            for (int i=0, j=m_playSequence.size() - 1; (i < sequencesFound[seqFndN].size()) && (j >= 0); ++i, j--)
                testSequence.addCard(m_playSequence[j]);
            sort(testSequence.begin(), testSequence.end());
            sort(sequencesFound[seqFndN].begin(), sequencesFound[seqFndN].end());
            testSequence.sortBySuit();
            sequencesFound[seqFndN].sortBySuit();
            if (testSequence == sequencesFound[seqFndN])
            {
                switch (m_rules.pairType(sequencesFound[seqFndN]))
                {
                case CribbageRules::PAIR:
                    m_pInterface->displayMessage(m_players.currentPlayer(), QString("%1 for a pair.").arg(CribbageRules::PAIR));
                    points += CribbageRules::PAIR;
                    break;
                case CribbageRules::PAIR_THREE:
                    m_pInterface->displayMessage(m_players.currentPlayer(), QString("%1 for a proil.").arg(CribbageRules::PAIR_THREE));
                    points += CribbageRules::PAIR_THREE;
                    break;
                case CribbageRules::PAIR_FOUR:
                    m_pInterface->displayMessage(m_players.currentPlayer(), QString("%1 for a double proil.").arg(CribbageRules::PAIR_FOUR));
                    points += CribbageRules::PAIR_FOUR;
                    break;
                case CribbageRules::PAIR_ERR:
                    break;
                }
            }
        }
    }

    //Are there any runs
    sequencesFound=cardProperties.runs();
    size=sequencesFound.size();
    for (int seqFound=0; seqFound < size; seqFound++)
    {
        testSequence.clear();
        // Add as many cards to our testSequence as we have in the sequence found.
        for (int i=0, j=m_playSequence.size() - 1; (i < sequencesFound[seqFound].size()) && (j >= 0); ++i, j--)
            testSequence.addCard(m_playSequence[j]);
        sort(testSequence.begin(), testSequence.end());
        sort(sequencesFound[seqFound].begin(), sequencesFound[seqFound].end());
        if (testSequence == sequencesFound[seqFound])
        {
            m_pInterface->displayMessage(m_players.currentPlayer(), QString("%1 for a run.").arg(sequencesFound[seqFound].size()));
            points+=sequencesFound[seqFound].size();
        }
    }

    // Did the player earn any points?
    if (points > 0)
        player.addPoint(points);
}

void Cribbage::roundOver()
{
    bool gameWon=false;

    // Set everyones turn off.
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].setTurn(false);
    restoreHands();
    if (! hasAPlayerWonTheRound())
    {
        score();
        m_pInterface->crib->setFaceUp(false);
        m_pInterface->player1Cards->setFaceUp(false);
        m_pInterface->updateTable();
    }
    m_pInterface->playerDiscards->clear(); // We have a delay setup, and so we need to clear the play sequence here.
    // See if anyone won the stake
    for (int i=0; i < m_players.size(); ++i)
        if (m_players[i].score() >= CribbageRules::WINNING_ROUND_SCORE)
        {
            m_players[i].incrementGamesWon();
            m_roundWinner = i;
            m_pInterface->displayMessage(m_players[i], "I won that round!");
            gameWon=true;
            // Check for skunk situation.
            if (m_players[(i + 1) % m_players.size()].score() <= CribbageRules::SKUNK)
                m_players[i].incrementGamesWon();
            if (m_players[(i + 1) % m_players.size()].score() <= CribbageRules::LURCHED)
                m_players[i].incrementGamesWon();
        }
    // See if our game is over
    if (m_rules.isGameOver(m_players))
        gameOver();
    else
        resetGame(gameWon);
}

bool Cribbage::hasAPlayerWonTheRound() const
{
    for (int i=0; i < m_players.size(); ++i)
        if (m_players[i].score() >= CribbageRules::WINNING_ROUND_SCORE)
            return true;
    return false;
}

void Cribbage::gameOver()
{
    m_pInterface->updateStakes(m_roundWinner);
    m_pInterface->updateTable();
    m_pInterface->gameWon();
}

void Cribbage::resetGame(bool roundOver)
{
    m_starter=Card();
    m_crib.clear();
    m_inGo=false;
    GameBase::resetGame(roundOver);
    if (roundOver)
        m_pInterface->updateStakes(m_roundWinner);
    else
        m_pInterface->resetGame();
    m_roundWinner = -1;
    deal();
}

void Cribbage::score()
{
    CardProperties playersCardProperties;
    CardSequence playersCards;

    // Counting and scoring occurs here. The hands are shown in strict order: nondealer, dealer's hand, crib.
    if (! m_players[0].isDealer())
        m_pInterface->player1Cards->setFaceUp(true);
    for (int i=0; i < m_players.size(); ++i)
    {
        if (i == m_dealerIndex)
            continue;

        // Add the starter card to the player's hand.
        playersCards=m_players[i].hand();
        playersCards.addCard(m_starter);

        //Calculate players hand
        playersCardProperties.setSequence(playersCards);
        scoring(m_players[i], playersCardProperties);

        //Display the summary
        if (m_cardSummary.size() > 0)
            m_pInterface->displayRoundSummary(m_players[i].name() + "'s Hand", m_cardSummary, m_messageSummary);
        else
            m_pInterface->promptMessage(m_players[i].name() + "'s Hand", "Has nineteen.");

        // Check to see if the player won.
        if (m_players[i].score() >= CribbageRules::WINNING_ROUND_SCORE)
            return;
    }

    //Calculate dealers hand
    if (m_players[0].isDealer())
        m_pInterface->player1Cards->setFaceUp(true);
    playersCards=m_players[m_dealerIndex].hand();
    playersCards.addCard(m_starter);
    playersCardProperties.setSequence(playersCards);
    scoring(m_players[m_dealerIndex], playersCardProperties);

    //Display the summary
    if (m_cardSummary.size() > 0)
        m_pInterface->displayRoundSummary(m_players[m_dealerIndex].name() + "'s Hand", m_cardSummary, m_messageSummary);
    else
        m_pInterface->promptMessage(m_players[m_dealerIndex].name() + "'s Hand", "Has nineteen.");

    //Calculate dealers crib
    m_scoringCrib=true;
    m_pInterface->crib->setFaceUp(m_scoringCrib);
    playersCards = m_crib;
    playersCards.addCard(m_starter);
    playersCardProperties.setSequence(playersCards);
    scoring(m_players[m_dealerIndex], playersCardProperties);
    m_scoringCrib=false;

    //Display the summary
    if (m_cardSummary.size() > 0)
        m_pInterface->displayRoundSummary(m_players[m_dealerIndex].name() + "'s Crib", m_cardSummary, m_messageSummary);
    else
        m_pInterface->promptMessage(m_players[m_dealerIndex].name() + "'s Crib", "Has nineteen.");
}

void Cribbage::scoring(Player &player, const CardProperties &cardProperties)
{
    int points=0;

    // Clear our records.
    m_cardSummary.clear();
    m_messageSummary.clear();

    // Tabulate Scores
    points+=scoreFifteen(cardProperties);
    points+=scorePairs(cardProperties);
    points+=scoreRuns(cardProperties);
    points+=scoreFlush(cardProperties);
    points+=scoreNobs(cardProperties);
    if (points > 0)
    {
        player.addPoint(points);
        m_cardSummary.push_back(CardSequence());
        m_messageSummary.push_back(QString("Total: %1").arg(points));
    }
    m_pInterface->updateTable();
}

int Cribbage::scoreFifteen(const CardProperties &cardProperties)
{
    vector<CardSequence> seq;

    // Find the combinations of 15.
    seq=cardProperties.summations(15, m_rules);

    // Inform the player about his 15s
    for (int i=0; i < static_cast<int>(seq.size()); ++i)
    {
        m_cardSummary.push_back(seq[i]);
        m_messageSummary.push_back(QString("%1 for fifteen").arg(CribbageRules::FIFTEEN));
    }
    //Give the number of fifteens found
    return seq.size() * CribbageRules::FIFTEEN;
}

int Cribbage::scorePairs(const CardProperties &cardProperties)
{
    vector<CardSequence> seq=cardProperties.pairs();
    int points=0;

    m_rules.createPairTypes(seq);
    // Inform the player about his pairs
    for (int i=0; i < static_cast<int>(seq.size()); ++i)
    {
        m_cardSummary.push_back(seq[i]);
        switch (seq[i].size())
        {
        case 2:
            m_messageSummary.push_back(QString("%1 for a pair").arg(CribbageRules::PAIR));
            points+=CribbageRules::PAIR;
            break;
        case 3:
            m_messageSummary.push_back(QString("%1 for a proil").arg(CribbageRules::PAIR_THREE));
            points+=CribbageRules::PAIR_THREE;
            break;
        case 4:
            m_messageSummary.push_back(QString("%1 for a double proil").arg(CribbageRules::PAIR_FOUR));
            points+=CribbageRules::PAIR_FOUR;
            break;
        }
    }
    return points;
}

int Cribbage::scoreRuns(const CardProperties &cardProperties)
{
    vector<CardSequence> seq=cardProperties.runs();
    int points=0;
    QString message;

    if (seq.size() == 0)
        return points;
    m_rules.createRunTypes(seq);
    m_cardSummary.push_back(seq.front());
    switch (m_rules.runType(cardProperties.runs()))
    {
    case CribbageRules::RUN:
        points+=seq.front().size() * CribbageRules::RUN;
        m_messageSummary.push_back(QString("%1 for a simple run").arg(seq.front().size() * CribbageRules::RUN));
        break;
    case CribbageRules::RUN_DOUBLE:
        points+=CribbageRules::RUN_DOUBLE;
        m_messageSummary.push_back(QString("%1 for a double run").arg(CribbageRules::RUN_DOUBLE));
        points-=removePairPoints(seq.front());
        break;
    case CribbageRules::RUN_DOUBLE_FOUR:
        points+=CribbageRules::RUN_DOUBLE_FOUR;
        m_messageSummary.push_back(QString("%1 for a double run of four").arg(CribbageRules::RUN_DOUBLE_FOUR));
        points-=removePairPoints(seq.front());
        break;
    case CribbageRules::RUN_TRIPLE:
        points+=CribbageRules::RUN_TRIPLE;
        m_messageSummary.push_back(QString("%1 for a triple run").arg(CribbageRules::RUN_TRIPLE));
        points-=removePairPoints(seq.front());
        break;
    case CribbageRules::RUN_QUADRUPLE:
        points+=CribbageRules::RUN_QUADRUPLE;
        m_messageSummary.push_back(QString("%1 for a quadruple run").arg(CribbageRules::RUN_QUADRUPLE));
        points-=removePairPoints(seq.front());
        break;
    case CribbageRules::RUN_ERR:
        //Player must not have had a run.
        break;
    }
    return points;
}

int Cribbage::removePairPoints(const CardSequence &run)
{
    CardSequence pair; //Is the pair in the run.
    bool foundPair;

    //Find our pair
    for (int i=0; i < (run.size()-1); ++i)
        if (run[i].rank() == run[i+1].rank())
        {
            if (! pair.hasCard(run[i]))
                pair.addCard(run[i]);
            pair.addCard(run[i+1]);
        }
    // remove pair from m_cardSummary and m_messageSummary
    for (int cardSummaryIndex=0; cardSummaryIndex < static_cast<int>(m_cardSummary.size()); ++cardSummaryIndex)
    {
        foundPair=false;
        // Exclude anything that couldn't be a pair.
        if ((m_cardSummary[cardSummaryIndex].size() != 2) && (m_cardSummary[cardSummaryIndex].size() != 3))
            continue;
        // Quad pairs need a special search
        if ((pair.size() == 2) || (pair.size() == 3))
        {
            // Is our pair in the current m_cardSummary?
            if (m_cardSummary[cardSummaryIndex].hasCard(pair[0]))
            {
                // Make sure it's a pair in the card summary
                if (m_cardSummary[cardSummaryIndex][0].rank() != m_cardSummary[cardSummaryIndex][1].rank())
                    continue;
                else
                    foundPair=true;
            }
        }
        else if (pair.size() == 4)
        {
            // Is our pair in the current m_cardSummary?
            if (m_cardSummary[cardSummaryIndex].hasCard(pair[0]))
            {
                // Make sure it's a pair in the card summary
                if (m_cardSummary[cardSummaryIndex][0].rank() != m_cardSummary[cardSummaryIndex][1].rank())
                    continue;
                else
                    foundPair=true;
            }
            // Is the other pair in the current m_cardSummary?
            if (m_cardSummary[cardSummaryIndex].hasCard(pair[2]))
            {
                // Make sure it's a pair in the card summary
                if (m_cardSummary[cardSummaryIndex][0].rank() != m_cardSummary[cardSummaryIndex][1].rank())
                    continue;
                else
                    foundPair=true;
            }
        }
        else
            throw KardsGTError("Cribbage", "removePairPoints", "Failed to handle pair type!");
        if (foundPair)
        {
            vector<CardSequence>::iterator cardIter=m_cardSummary.begin();
            vector<QString>::iterator messageIter=m_messageSummary.begin();
            for (int k=0; k < cardSummaryIndex; ++k)
            {
                cardIter++;
                messageIter++;
            }
            m_cardSummary.erase(cardIter);
            m_messageSummary.erase(messageIter);
            --cardSummaryIndex;
        }
    }
    if (pair.size() == 2)
        return CribbageRules::PAIR;
    else if (pair.size() == 3)
        return CribbageRules::PAIR_THREE;
    else if (pair.size() == 4)
        return CribbageRules::PAIR * 2;
    else
        return 0;
}

int Cribbage::scoreFlush(const CardProperties &cardProperties)
{
    int points=0;
    vector<CardSequence> seq=cardProperties.flushes();
    QString message;

    // Inform the player about his flushes
    if (! seq.empty())
        switch (seq.back().size())
        {
        case 4:
            // Don't allow a flush that has the starter card as it's fourth card.
            if (seq.back().hasCard(m_starter) || m_scoringCrib)
                return 0;
            points=CribbageRules::FLUSH_FOUR;
            m_cardSummary.push_back(seq.front());
            m_messageSummary.push_back(QString("%1 for a flush of four").arg(CribbageRules::FLUSH_FOUR));
            break;
        case 5:
            points=CribbageRules::FLUSH_FIVE;
            m_cardSummary.push_back(seq.front());
            m_messageSummary.push_back(QString("%1 for a flush of five").arg(CribbageRules::FLUSH_FIVE));
            break;
        }
    return points;
}

int Cribbage::scoreNobs(const CardProperties &cardProperties)
{
    CardSequence hand = cardProperties.sequence();

    // Find nobs
    for (int i=0, size = hand.size() - 1; i < size; ++i)
        if ((hand[i].rank() == Card::JACK) && (hand[i] != m_starter))
            if (hand[i].suit() == m_starter.suit())
            {
                CardSequence nobs;

                nobs.addCard(hand[i]);
                m_cardSummary.push_back(nobs);
                m_messageSummary.push_back(QString("%1 for his nobs").arg(CribbageRules::NOBS));
                return CribbageRules::NOBS;
            }
    return 0;
}
