/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef PLAYERLIST_H
#define PLAYERLIST_H

#include <vector>
using std::vector;
#include <iostream>
using std::ostream;

class Player;
class QTextStream;

/**
 * This is the list of all players in a game.
 *
 * @note This class does NOT handle the memory management of any players!
 * @author John Schneiderman
 */
class PlayerList
{
public:
    /**
     * Default constructor.
     */
    PlayerList();
    /**
     * Inserts a player into the list.
     * @param player is the player to place in the list.
     */
    void insert(Player &player);
    /**
     * Removes a player from the list.
     * @param player is the player to remove from the list.
     * @return true if successfully removed, false elsewise.
     */
    bool remove(const Player &player);
    /**
     * Increments to the next player.
     * @note The player list is circular.
     * @return the new current player.
     */
    Player& nextPlayer();
    /**
     * Increments to the next player who still has cards in his hand.
     * @note The list is circular.
     * @throw KardsGTError If there are not any players in the list with cards.
     * @return the new current player.
     */
    Player& nextNonEmptyPlayer();
    /**
     * Accessor to the dealer player.
     * @return the current dealer.
     * @throw KardsGTError if no player is found to be the dealer.
     */
    Player& dealer();
    /**
     * Accessor to the current player.
     * @return the current player.
     */
    Player& currentPlayer();
    /**
     * Accessor to the current player's index.
     * @return the current player's index.
     */
    int currentPlayerIndex() const;
    /**
     * Accessor to the player that is to the right of the current player.
     * @return the right player.
     */
    Player& rightPlayer();
    /**
     * Accessor to the player that is to the left of the current player.
     * @return the left player.
     */
    Player& leftPlayer();
    /**
     * Accessor to the player that is to the right of the current player with cards.
     * @throw KardsGTError If there are not any players in the list with cards.
     * @return the right player with cards.
     */
    Player& rightPlayerWithCards();
    /**
     * Accessor to the player that is to the left of the current player with cards.
     * @throw KardsGTError If there are not any players in the list with cards.
     * @return the left player with cards.
     */
    Player& leftPlayerWithCards();
    /**
     * Sets the current player.
     * @param player is the player to set as the new current player.
     * @throw KardsGTError if player does not exist in the list of players.
     */
    void setCurrentPlayer(const Player &player);
    /**
     * This is the size of the players list.
     * @return how many players are in the list.
     */
    int size() const;
    /**
     * Random accsesor to the list of players.
     * @param index is the player to retrieve.
     * @throw KardsGTError if the index is out of range.
     * @return the player requested.
     */
    Player& operator[](int index);
    /**
     * Random accsesor to the list of players.
     * @param index is the player to retrieve.
     * @throw KardsGTError if the index is out of range.
     * @return the player requested as a point to a const.
     */
    const Player& operator[](int index) const;
    /**
     * Clears the list of players.
     */
    void clear();
    /**
     * The stream insertion for the player list.
     * @param out is the stream to insert the player's list into.
     * @param playerList is the list of players to place into the stream.
     * @return out.
     */
    friend QTextStream& operator<<(QTextStream &out, const PlayerList &playerList);
    /**
     * The stream extraction for the player list.
     * @param in is the stream to extract the player's list from.
     * @param playerList is the list of players to fill with the data from the stream.
     * @return in.
     */
    friend QTextStream& operator>>(QTextStream &in, PlayerList &playerList);
    /**
     * The stream insertion for the player list.
     * @param out is the stream to insert the player's list into.
     * @param playerList is the list of players to place into the stream.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const PlayerList &playerList);
    /**
     * Determines if the player's list is empty.
     * @return true if there are no players in the list, false if there are.
     */
    bool isEmpty() const;
    /**
     * Swaps two player locations in the list.
     * @param left is the left player to be swapped into the right position.
     * @param right is the right player to be swapped into the left position.
     */
    void swap(int left, int right);

private:
    /// @param m_playerList is an array that points to each of the players in the game.
    vector<Player *> m_playerList;
    /// @param m_current is the current player we're focusing on.
    int m_current;
};
#endif
