/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "carddeck.h"

#include <qstring.h>
#include <cmath>
#include <algorithm>

CardDeck::CardDeck(): CardPile(), m_removedRanks()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    createDeck();
}

CardDeck::CardDeck(vector<Card::Rank> ranks): CardPile()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    m_removedRanks = ranks;
    createDeck();
}

void CardDeck::shuffle()
{
    // See if the deck is missing some cards
    if (m_pile.size() != DECK_SIZE)
    {
        CardSequence missingCards = findMissingCards();

        // Shuffle the missing cards, because no one sorts the cards before they put them back in the deck
        random_shuffle(missingCards.begin(), missingCards.end());
        // Add any missing cards
        for (int i=0; i < missingCards.size(); ++i)
            m_pile.addCard(missingCards[i]);
    }

    // Shuffle the deck
    random_shuffle(m_pile.begin(), m_pile.end());
}

Card CardDeck::dealCard()
{
    return removeCard();
}

void CardDeck::cutDeck()
{
    // Ensure that we at least select GUARANTEE_CUT_SIZE cards.
    int newTopCard=(rand() % m_pile.size()) + GUARANTEE_CUT_SIZE;
    int pileSize=m_pile.size();
    CardDeck leftDeck, rightDeck;

    // Clear the decks since we want to use them as holders.
    leftDeck.clear();
    rightDeck.clear();
    // Cut the deck
    for (int cardIndex=0; cardIndex < pileSize; cardIndex++)
        if (cardIndex < newTopCard)
            leftDeck.addCard(m_pile[cardIndex]);
        else
            rightDeck.addCard(m_pile[cardIndex]);

    // Combine the two decks.
    *this=rightDeck;
    *this+=leftDeck;
}

void CardDeck::createDeck()
{
    Card card;

    m_pile.clear();
    for (int i=0; i < Card::NUMBER_OF_SUITS; i++)
        for (int j=0; j < Card::NUMBER_OF_RANKS; j++)
        {
            card = Card(static_cast<Card::Rank>(j), static_cast<Card::Suit>(i));
            if (allowedCard(card))
                m_pile.addCard(card);
        }
}

CardSequence CardDeck::findMissingCards() const
{
    CardSequence missingCards;
    Card testCard;

    // Find the missing cards
    for (int i=0; i < Card::NUMBER_OF_SUITS; i++)
        for (int j=0; j < Card::NUMBER_OF_RANKS; j++)
        {
            testCard = Card(static_cast<Card::Rank>(j), static_cast<Card::Suit>(i));
            if (allowedCard(testCard) && (! m_pile.hasCard(testCard)))
                missingCards.addCard(testCard);
        }
    return missingCards;
}

void CardDeck::removeCards(Card::Rank rank)
{
    m_removedRanks.push_back(rank);
    m_pile.clear();
    createDeck();
}

bool CardDeck::allowedCard(const Card &card) const
{
    for (int index = 0, size = m_removedRanks.size(); index < size; ++index)
        if (card.rank() == m_removedRanks[index])
            return false;
    return true;
}
