/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "norman.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// The games Norman knows how to play
#include "normancribbageai.h"
#include "normanheartsai.h"
#include "normanspadesai.h"

Norman::Norman(): AIBase()
{}

Norman::~Norman()
{}

CardSequence Norman::whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Norman::bidAt(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::skill_Level Norman::skillLevel(const QString& game) const
{
    if (game == "spades")
        return Player::Novice;
    else if (game == "cribbage")
        return Player::Amateur;
    else if (game == "hearts")
        return Player::Professional;
    else if (game == "old maid")
        return Player::Master;
    else
        return Player::Newb;
}

QString Norman::name() const
{
    return "Norman";
}

CardSequence Norman::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "cribbageDealerCrib")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.myCrib();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        NormanCribbageAI ai(playSequence, rules, hand);
        return ai.opponentsCrib();
    }
    throw KardsGTError("Norman", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Norman::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards();
    }
    else if (game == "spadesExchangeCards")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Norman", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Norman::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        NormanHeartsAI ai(playSequence, rules, hand);
        return ai.selectCards();
    }
    else if (game == "heartsPassPhase")
    {
        NormanHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Norman", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Norman::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        NormanSpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Norman", "handleSpadesBid", "Failed to determine game type or phase.");
}
