/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "basicgamestrategies.h"
#include "cardproperties.h"
#include "rulebase.h"
#include "kardsgterror.h"

BasicGameStrategies::BasicGameStrategies(const RuleBase &rules)
{
    m_pRules = &rules;
}

BasicGameStrategies::~BasicGameStrategies()
{}

Card BasicGameStrategies::highestCard(const CardSequence &sequence) const
{
    int highestCardIndex = 0;

    if (sequence.isEmpty())
        return Card();

    for (int index = 1; index < sequence.size(); ++index)
        if (m_pRules->rankValue(sequence[index]) > m_pRules->rankValue(sequence[highestCardIndex]))
            highestCardIndex = index;
    return sequence[highestCardIndex];
}

Card BasicGameStrategies::highestNonTrumpCard(const CardSequence &sequence, Card::suit_t trumpSuit) const
{
    CardProperties properties(sequence);
    CardSequence testSequence, spades, hearts, clubs, diamonds;

    // Get the highest sequence for each suit
    if (trumpSuit != Card::SPADES)
        spades = highestCard(properties.suits(Card::SPADES));
    if (trumpSuit != Card::HEARTS)
        hearts = highestCard(properties.suits(Card::HEARTS));
    if (trumpSuit != Card::DIAMONDS)
        diamonds = highestCard(properties.suits(Card::DIAMONDS));
    if (trumpSuit != Card::CLUBS)
        clubs = highestCard(properties.suits(Card::CLUBS));
    // Add the highest sequence of each suit to our test sequence
    if (! spades.isEmpty())
        testSequence += spades;
    if (! hearts.isEmpty())
        testSequence += hearts;
    if (! diamonds.isEmpty())
        testSequence += diamonds;
    if (! clubs.isEmpty())
        testSequence += clubs;
    // Find the highest card in testSeqeuence
    return highestCard(testSequence);
}

Card BasicGameStrategies::highestCardOfSuit(const CardSequence &sequence, Card::suit_t suit) const
{
    CardProperties properties(sequence);

    return highestCard(properties.suits(suit));
}

Card BasicGameStrategies::nextHighestCard(const CardSequence &sequence, const Card &highCard) const
{
    CardSequence testSeq = sequence;

    // Remove all cards that are euqal in rank to our high card or higher.
    for (int index = 0; index < testSeq.size(); ++index)
        if (m_pRules->rankValue(testSeq[index]) >= m_pRules->rankValue(highCard))
            testSeq.removeCard(testSeq[index]);
    testSeq = highestCard(testSeq);
    if (testSeq.isEmpty())
        return highCard;
    else
        return testSeq.front();
}

Card BasicGameStrategies::nextHighestCardOfSuit(const CardSequence &sequence, const Card &highCard, Card::suit_t suit) const
{
    CardSequence testSeq = sequence;

    // Remove all cards that are euqal in rank to our high card or higher.
    for (int index = 0; index < testSeq.size(); ++index)
        if (m_pRules->rankValue(testSeq[index]) >= m_pRules->rankValue(highCard))
            testSeq.removeCard(testSeq[index]);
    CardProperties properties(testSeq);
    testSeq = highestCard(properties.suits(suit));
    if (testSeq.isEmpty())
        return highCard;
    else
        return testSeq.front();
}

Card BasicGameStrategies::lowestCard(const CardSequence &sequence) const
{
    int lowestCardIndex = 0;

    if (sequence.isEmpty())
        return Card();

    for (int index=1; index < sequence.size(); ++index)
        if (m_pRules->rankValue(sequence[index]) < m_pRules->rankValue(sequence[lowestCardIndex]))
            lowestCardIndex = index;
    return sequence[lowestCardIndex];
}

Card BasicGameStrategies::lowestNonTrumpCard(const CardSequence &sequence, Card::suit_t trumpSuit) const
{
    CardProperties properties(sequence);
    CardSequence testSequence, spades, hearts, clubs, diamonds;

    // Get the lowest sequence for each suit
    if (trumpSuit != Card::SPADES)
        spades = lowestCard(properties.suits(Card::SPADES));
    if (trumpSuit != Card::HEARTS)
        hearts = lowestCard(properties.suits(Card::HEARTS));
    if (trumpSuit != Card::DIAMONDS)
        diamonds = lowestCard(properties.suits(Card::DIAMONDS));
    if (trumpSuit != Card::CLUBS)
        clubs = lowestCard(properties.suits(Card::CLUBS));
    // Add the lowest sequence of each suit to our test sequence
    if (! spades.isEmpty())
        testSequence += spades;
    if (! hearts.isEmpty())
        testSequence += hearts;
    if (! diamonds.isEmpty())
        testSequence += diamonds;
    if (! clubs.isEmpty())
        testSequence += clubs;
    // Find the lowest card in testSeqeuence
    return lowestCard(testSequence);
}

Card BasicGameStrategies::lowestCardOfSuit(const CardSequence &sequence, Card::suit_t suit) const
{
    CardProperties properties(sequence);

    return lowestCard(properties.suits(suit));
}

CardSequence BasicGameStrategies::selectCards(int numberOfCards, const CardSequence &hand, const CardSequence &playSequence) const
{
    CardSequence cards;
    Player player;

    // Create the necessary player information for the legal play method
    for (int i=0; i < hand.size(); ++i)
        player.hand().addCard(hand[i]);
    for (int cardIndex=0; (cardIndex < hand.size()) && (cards.size() < numberOfCards); ++cardIndex)
        if (m_pRules->isLegalPlay(playSequence, hand[cardIndex], player))
            cards.addCard(hand[cardIndex]);
    if (cards.size() == numberOfCards)
        return cards;
    else
        throw KardsGTError("BasicGameStrategies", "selectCards", "Could not find enough cards to play!");
}

CardSequence BasicGameStrategies::randomCards(int numberOfCards, const CardSequence &hand, const CardSequence &playSequence) const
{
    CardSequence cards;
    Player player;
    int cardIndex;

    // Create the necessary player information for the legal play method
    for (int i=0; i < hand.size(); ++i)
        player.hand().addCard(hand[i]);
    // Find our cards to play
    while (cards.size() != numberOfCards)
    {
        cardIndex=rand() % hand.size();
        if (! cards.hasCard(hand[cardIndex]))
            if (m_pRules->isLegalPlay(playSequence, hand[cardIndex], player))
                cards.addCard(hand[cardIndex]);
    }
    return cards;
}

CardSequence BasicGameStrategies::selectCardsWithNoLegalChecks(int numberOfCards, const CardSequence &hand) const
{
    CardSequence cards;

    for (int cardIndex=0; cardIndex < numberOfCards; ++cardIndex)
        cards.addCard(hand[cardIndex]);
    return cards;
}

CardSequence BasicGameStrategies::randomCardsWithNoLegalChecks(int numberOfCards, const CardSequence &hand) const
{
    CardSequence cards;
    int cardIndex;

    while (cards.size() != numberOfCards)
    {
        cardIndex=rand() % hand.size();
        if (! cards.hasCard(hand[cardIndex]))
            cards.addCard(hand[cardIndex]);
    }
    return cards;
}

int BasicGameStrategies::randomBid(int minimum, int maximum) const
{
    return (rand() % maximum) + minimum;
}
