/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef WAR_RULES_H_
#define WAR_RULES_H_

#include "rulebase.h"
#include "card.h"
#include "cardsequence.h"
#include "playerlist.h"
#include "player.h"

/**
 * This is the rules for a game of War.
 *
 * @author John Schneiderman
 */
class WarRules: public RuleBase
{
public:
    /**
     * These are the public game constants for the rules of war.
     * @param CARDS_TO_WAR_PILE are the number of cards to place in a war pile.
     */
    enum PublicConstants { CARDS_TO_WAR_PILE = 3 };

    /**
     * Default constructor.
     */
    WarRules();
    /**
     * Gives the maximum number of players allowed in a game of war.
     * @return how many can play the game.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players allowed in a game of war.
     * @return how many must play the game.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the score need by a player to win.
     * @return the score needed to win.
     */
    virtual int winningGameScore() const;
    /**
     * Gives how many cards to deal to each player in the game.
     * @return the number of cards to deal.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers=-1) const;
    /**
     * Gives how much value a particular card has.
     * @param card is the card whose value we want to know.
     * @return the value of the card.
     */
    virtual int cardValue(const Card &card) const;
    /**
     * Determines if the card the player wants to play is a legal play.
     * @param sequence is the sequence of cards already played.
     * @param cardPlayed is the card the player wants to play.
     * @param player is the player playing the card.
     * @return true if cardPlayed is a legal play, false if it is not.
     */
    virtual bool isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &player=Player()) const;
    /**
     * Determines if the game is over.
     * @param players are all the players in the game.
     * @return true if the game is over, false if it is not.
     */
    virtual bool isGameOver(const PlayerList &players) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual bool isRoundOver(const PlayerList &) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual bool isPhaseOver(const PlayerList &, const CardSequence &) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual int rankValue(const Card &card) const;

private:
    /**
     * These are the constants used through-out the rules of war.
     * @param MAXIMUM_NUMBER_OF_PLAYERS is the maximum number of players allowed in a game of war.
     * @param MINIMUM_NUMBER_OF_PLAYERS is the minimum number of players needed in a game of war.
     * @param WINNING_GAME_SCORE is the score needed to win the game.
     * @param NUMBER_OF_CARDS_TO_DEAL is the number of cards to deal to each player.
     */
    enum PrivateConstants { MAXIMUM_NUMBER_OF_PLAYERS = 2, MINIMUM_NUMBER_OF_PLAYERS = 2, WINNING_GAME_SCORE = 3, NUMBER_OF_CARDS_TO_DEAL = 26 };
    /// These are the card values for each card. In war they are worth their face value. Starting with ace as the lowest.
    enum CardValues { ACE=1, TWO=2, THREE=3, FOUR=4, FIVE=5, SIX=6, SEVEN=7, EIGHT=8, NINE=9, TEN=10, JACK=11, QUEEN=12, KING=13, RANK_ERR=-1 };
};
#endif
