/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef USERPROFILE_H
#define USERPROFILE_H

#include <qstring.h>
#include <vector>
using std::vector;
using std::pair;

/**
 * This is the database of user profiles.
 *
 * When a database is loaded it contains the information on what profiles we are tracking,
 * and how well they're playing each game. The database structure is simple. It has a game
 * directory located in the base path. Inside the game directory is the user profile list
 * and directories for each of the users. Inside each of the user directories are files
 * named for each game available. The game files are named after the game it's tracking.
 * Inside the game file is just the games won and the games lost, each entry on their own
 * line. There is also a file that holds each of the users image files and settings.
 * The layout of the file is: \n
 * label setting
 *
 * @todo Add code to handle multiple mood images for the user
 *
 * @author John Schneiderman
 */
class UserProfileDatabase
{
public:
    /**
     * The default constructor.
     * @param path is the location to the current user profile database or to where we want to create the database.
     * @throw KardsGTError if path is empty.
     */
    UserProfileDatabase(const QString &path = ".");
    /**
     * The default destructor.
     */
    ~UserProfileDatabase();
    /**
     * An accessor to all the users in our database.
     * @return the listing of all the users in our database.
     */
    vector<QString> users() const;
    /**
     * Sets the users listed in our database.
     * @param users is the list of users.
     * @note If the size of users is greater than our current list we assume a new profile is to be added.
     * @note If the size of users is less than our current list we assume a profile is to be removed.
     * @note All database changes are written immediately to disk.
     */
    void setUsers(const vector<QString> &users);
    /**
     * An accessor to a users game statistics.
     * @param user is the name of the user in our database.
     * @param game is the name of the game we want the statistics for.
     * @return A pair whose first value is the number of games won, and whose second value is the number of games played.
     */
    pair<int, int> userStatistics(const QString &user, const QString &game) const;
    /**
     * Updates the game statistics for a user, if the game statistics file doesn't exist one is created.
     * @note If the game statistic file is not found, one will be created.
     * @param user is the name of the user in our database.
     * @param game is the name of the game we want to update the statistics for.
     * @param won is true if the user has won the game, false if he has lost.
     * @throw KardsGTError if we cannot create the game profile.
     */
    void setUserGamesWon(const QString &user, const QString &game, bool won);
    /**
     * Loads the user profile database, if the database cannot be found a blank database is created.
     * @throw KardsGTError if cannot create the database.
     * @throw KardsGTError if we cannot read the database.
     */
    void loadUserProfileDatabase();
    /**
     * Saves the user profile database.
     * @throw KardsGTError if we cannot save the database.
     */
    void saveUserProfileDatabase();
    /**
     * This retrieves the image to use for the user.
     * @note The information is found in the user settings file.
     * @param user is the user we're getting the image for.
     * @param mood is the mood of the image we're getting.
     * @return the stored player's image filename, if not found player-male.png is returned.
     */
    QString userImage(const QString &user, const QString &mood = "neutral") const;
    /**
     * This retrieves the path to the directory that has the card images.
     * @note The information is pulled from the user settings file.
     * @return the stored path to the directory, if not found default is returned.
     */
    QString pathToCardImages(const QString &user) const;
    /**
     * This retrieves the image to use for the back of the cards.
     * @note The information is pulled from the user settings file.
     * @return the stored filename for the image to use on the back of the card, if not found "back00.png" is returned.
     */
    QString pathToCardBackImage(const QString &user) const;
    /**
     * This saves a users image settings.
     * @note The information is put into the user settings file.
     * @param user is the user to save the settings for.
     * @param settings are all the settings to place into the user settings file. The first pair is the label and the second pair is the setting value.
     * @throw KardsGTError if we cannot save the settings file.
     */
    void setUserSettings(const QString &user, const vector<pair<QString, QString> > &settings);

protected:
    /**
     * Adds the new user to our user database.
     * @param users is the new user list.
     */
    void addUser(const vector<QString> &users);
    /**
     * Removes the missing user from our database.
     * @param users is the new user list.
     */
    void removeUser(const vector<QString> &users);

private:
    /// @param m_basePath is the base path to our database.
    QString m_basePath;
    /// @param m_users are the users in our database.
    vector<QString> m_users;
    /// @param INVALID_USER is the value that represents an invalid user.
    static const QString INVALID_USER;
    /// @param MAKE_DIRECTORY is the system command needed to make a single directory.
    static const QString MAKE_DIRECTORY;
    /// @param MAKE_DIRECTORY_PATH is the system command needed to make all the directories necessary for a valid path.
    static const QString MAKE_DIRECTORY_PATH;
    /// @param CHANGE_DIRECTORY is the system command needed to change into a new directory.
    static const QString CHANGE_DIRECTORY;
    /// @param REMOVE_DIRECTORY is the system command needed to remove all files and directories.
    static const QString REMOVE_DIRECTORY;
    /// @param DIRECTORY_MARKER is the character used to separate directories in the command line.
    static const QString DIRECTORY_MARKER;
    /// @param GAME_DIRECTORY is the directory for where the game database is stored.
    static const QString GAME_DIRECTORY;
    /// @param PROFILE_DIRECTORY is the directory for where we want to store our profiles.
    static const QString PROFILE_DIRECTORY;
    /// @param STATISTICS_EXTENSION is the extension to use on the statistics files.
    static const QString STATISTICS_EXTENSION;
    /// @param PROFILE_DATABASE_NAME is the name of our profile database.
    static const QString PROFILE_DATABASE_NAME;
    /// @param USER_SETTINGS_EXTENSION is the extension to use on the user settings
    static const QString USER_SETTINGS_EXTENSION;
};
#endif
