/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "spadesinterface.h"
#include "kardsequence.h"
#include "kardplayer.h"
#include "kardplaysequence.h"
#include "kardselection.h"
#include "kardsgterror.h"
#include "bidrequest.h"
#include "kard.h"
#include "spadesplayer.h"
#include "userprofiledatabase.h"

#include <qmessagebox.h>

SpadesInterface::SpadesInterface(UserProfileDatabase &profileDatabase, QWidget *parent, const char *name): SpadesInterfaceBase(parent, name), Spades(this, profileDatabase)
{
    m_pUserProfile = &profileDatabase;
    setBackgroundColor(darkGreen);
    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
    connect(player3Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player3CardPlayed(Kard &)));
    connect(player4Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player4CardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved()), this, SLOT(player1CardMoved()));
    connect(player2Cards, SIGNAL(kardMoved()), this, SLOT(player2CardMoved()));
    connect(player3Cards, SIGNAL(kardMoved()), this, SLOT(player3CardMoved()));
    connect(player4Cards, SIGNAL(kardMoved()), this, SLOT(player4CardMoved()));

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
    playSequence->setFaceUp(true);
}

SpadesInterface::~SpadesInterface()
{}

void SpadesInterface::updateTable()
{
    // Set the card back images
    player1Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player2Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player3Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player4Cards->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()));
    playSequence->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()));

    // Set the card front path
    player1Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player2Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player3Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player4Cards->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()));
    playSequence->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()));

    // Update player hands
    player1Cards->setCardSequence(m_players[Spades::PLAYER_ONE_INDEX].hand());
    player2Cards->setCardSequence(m_players[Spades::PLAYER_TWO_INDEX].hand());
    player3Cards->setCardSequence(m_players[Spades::PLAYER_THREE_INDEX].hand());
    player4Cards->setCardSequence(m_players[Spades::PLAYER_FOUR_INDEX].hand());
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        playSequence->setCardSequence(m_playSequence, m_playerCardOrderIndexes.front());

    // Update table info
    // Player 1
    player1->setPlayerImage(m_players[Spades::PLAYER_ONE_INDEX].name().lower() + ".png");
    player1->setDealer(m_players[Spades::PLAYER_ONE_INDEX].isDealer());
    player1->setTurn(m_players[Spades::PLAYER_ONE_INDEX].isTurn());
    player1->setLevel(m_players[Spades::PLAYER_ONE_INDEX].level());
    // player 2
    player2->setPlayerImage(m_players[Spades::PLAYER_TWO_INDEX].name().lower() + ".png");
    player2->setDealer(m_players[Spades::PLAYER_TWO_INDEX].isDealer());
    player2->setTurn(m_players[Spades::PLAYER_TWO_INDEX].isTurn());
    player2->setLevel(m_players[Spades::PLAYER_TWO_INDEX].level());
    // player 3
    player3->setPlayerImage(m_pUserProfile->userImage(m_players[Spades::PLAYER_THREE_INDEX].name()));
    player3->setDealer(m_players[Spades::PLAYER_THREE_INDEX].isDealer());
    player3->setTurn(m_players[Spades::PLAYER_THREE_INDEX].isTurn());
    player3->setLevel(m_players[Spades::PLAYER_THREE_INDEX].level());
    // player 4
    player4->setPlayerImage(m_players[Spades::PLAYER_FOUR_INDEX].name().lower() + ".png");
    player4->setDealer(m_players[Spades::PLAYER_FOUR_INDEX].isDealer());
    player4->setTurn(m_players[Spades::PLAYER_FOUR_INDEX].isTurn());
    player4->setLevel(m_players[Spades::PLAYER_FOUR_INDEX].level());
}

bool SpadesInterface::bidDouble()
{
    int responce = QMessageBox::question(this, "Pre-Look Bid", "Would you like to bid double nil?", QMessageBox::Yes, QMessageBox::No, QMessageBox::NoButton);

    if (responce == QMessageBox::Yes)
        return true;
    else
        return false;
}

CardSequence SpadesInterface::selectExchangeCards(const Player &player, const Player &partner)
{
    KardSelection passSelection(QString("Select %1 cards to exchange with %2").arg(SpadesRules::CARDS_TO_EXCHANGE).arg(partner.name()), SpadesRules::CARDS_TO_EXCHANGE, player.hand(), m_pUserProfile->pathToCardImages(m_players[Spades::PLAYER_THREE_INDEX].name()), m_pUserProfile->pathToCardBackImage(m_players[Spades::PLAYER_THREE_INDEX].name()), this, "exchange cards");

    if (passSelection.exec())
        return passSelection.selection();
    else
        return selectExchangeCards(player, partner);
}

void SpadesInterface::showCards(bool show, Spades::PlayerIndexes index)
{
    switch (index)
    {
    case Spades::PLAYER_ONE_INDEX:
        player1Cards->setFaceUp(show);
        break;
    case Spades::PLAYER_TWO_INDEX:
        player2Cards->setFaceUp(show);
        break;
    case Spades::PLAYER_THREE_INDEX:
        player3Cards->setFaceUp(show);
        break;
    case Spades::PLAYER_FOUR_INDEX:
        player4Cards->setFaceUp(show);
        break;
    default:
        throw KardsGTError("SpadesInterface", "showCards", "Failed to find player index.");
    }
}

int SpadesInterface::bid()
{
    BidRequest bidRequest("Contract bid", SpadesRules::MINIMUM_BID, SpadesRules::MAXIMUM_BID, "Nil", this, "bid");

    if (bidRequest.exec())
        return bidRequest.bid();
    else
        return bid();
}

void SpadesInterface::displayBid(Spades::PlayerIndexes index, int bid)
{
    QString formattedBid;

    if (bid == SpadesRules::DOUBLE_NIL)
        formattedBid = "double nil";
    else if (bid == SpadesRules::NIL)
        formattedBid = "nil";
    else
        formattedBid = QString("%1").arg(bid);
    switch (index)
    {
    case Spades::PLAYER_ONE_INDEX:
        player1->setCaption(QString("I bid " + formattedBid + " tricks."), 12000);
        break;
    case Spades::PLAYER_TWO_INDEX:
        player2->setCaption(QString("I bid " + formattedBid + " tricks."), 12000);
        break;
    case Spades::PLAYER_THREE_INDEX:
        player3->setCaption(QString("I bid " + formattedBid + " tricks."), 12000);
        break;
    case Spades::PLAYER_FOUR_INDEX:
        player4->setCaption(QString("I bid " + formattedBid + " tricks."), 12000);
        break;
    default:
        throw KardsGTError("SpadesInterface", "displayBid", "Failed to find player index.");
    }
    updateTable();
}

void SpadesInterface::displayMessage(const Player &player, const QString &message)
{
    if (player.name() == m_players[Spades::PLAYER_ONE_INDEX].name())
        player1->setCaption(message);
    else if (player.name() == m_players[Spades::PLAYER_TWO_INDEX].name())
        player2->setCaption(message);
    else if (player.name() == m_players[Spades::PLAYER_THREE_INDEX].name())
        player3->setCaption(message);
    else if (player.name() == m_players[Spades::PLAYER_FOUR_INDEX].name())
        player4->setCaption(message);
    else
        throw KardsGTError("SpadesInterface", "displayMessage", "Player not found!");
}

void SpadesInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
}

void SpadesInterface::gameWon()
{
    QString finalScores;

    // Create the final score messages.
    finalScores = QString("These are the final scores:\n %1-%2 Team: %3 points\n %4-%5 Team: %6 points").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name()).arg(m_players[Spades::PLAYER_ONE_INDEX].score()).arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name()).arg(m_players[Spades::PLAYER_TWO_INDEX].score());
    // Find the winner
    switch (teamWhoWon())
    {
    case Spades::TEAM_ONE:
        promptMessage("Game Over", QString("Team " + m_players[Spades::PLAYER_ONE_INDEX].name() + "-" + m_players[Spades::PLAYER_THREE_INDEX].name() + " won the game.\n\n" + finalScores));
        m_pProfileDatabase->setUserGamesWon(m_players[Spades::PLAYER_THREE_INDEX].name(), "spades", true);
        break;
    case Spades::TEAM_TWO:
        promptMessage("Game Over", QString("Team " + m_players[Spades::PLAYER_TWO_INDEX].name() + "-" + m_players[Spades::PLAYER_FOUR_INDEX].name() + " won the game.\n\n" + finalScores));
        m_pProfileDatabase->setUserGamesWon(m_players[Spades::PLAYER_THREE_INDEX].name(), "spades", false);
        break;
    default:
        throw KardsGTError("SpadesInterface", "gameWon", "No one won?");
    }
}

void SpadesInterface::player1CardPlayed(Kard &kard)
{
    kardPlayed(Spades::PLAYER_ONE_INDEX, kard);
}

void SpadesInterface::player2CardPlayed(Kard &kard)
{
    kardPlayed(Spades::PLAYER_TWO_INDEX, kard);
}

void SpadesInterface::player3CardPlayed(Kard &kard)
{
    kardPlayed(Spades::PLAYER_THREE_INDEX, kard);
}

void SpadesInterface::player4CardPlayed(Kard &kard)
{
    kardPlayed(Spades::PLAYER_FOUR_INDEX, kard);
}

void SpadesInterface::displayScores()
{
    promptMessage("Scores", QString("These are the scores:\n %1-%2 Team: %3 points\n %4-%5 Team: %6 points").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name()).arg(m_players[Spades::PLAYER_ONE_INDEX].score()).arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name()).arg(m_players[Spades::PLAYER_TWO_INDEX].score()));
}

void SpadesInterface::displayTricks()
{
    int size=m_players.size();

    for (int playerIndex=0; playerIndex < size; ++playerIndex)
        displayMessage(m_players[playerIndex], QString("I have %1 tricks.").arg(m_tricksWon[playerIndex]));
}

void SpadesInterface::displayContracts()
{
    pair<int, int> contract;

    contract.first = calculateContract(pair<int, int>(Spades::PLAYER_ONE_INDEX, Spades::PLAYER_THREE_INDEX));
    contract.second = calculateContract(pair<int, int>(Spades::PLAYER_TWO_INDEX, Spades::PLAYER_FOUR_INDEX));
    promptMessage("Contract", QString("These are the contracts:\n %1-%2 Team: %3 tricks\n %4-%5 Team: %6 tricks").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name()).arg(contract.first).arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name()).arg(contract.second));
}

void SpadesInterface::kardPlayed(PlayerIndexes playerIndex, Kard &kard)
{
    if (m_players[playerIndex].isTurn())
    {
        if (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[playerIndex]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
        else
        {
            promptMessage("Warning", QString("%1 is not a legal play!").arg(kard.card().toString()));
            kard.setSelected(false);
        }
    }
    else
    {
        promptMessage("Warning", QString("%1 it is not your turn.").arg(m_players[playerIndex].name()));
        kard.setSelected(false);
    }
}

void SpadesInterface::player1CardMoved()
{
    CardSequence playersHand = player1Cards->cardSequence();

    m_players[Spades::PLAYER_ONE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Spades::PLAYER_ONE_INDEX].hand().addCard(playersHand[index]);
}

void SpadesInterface::player2CardMoved()
{
    CardSequence playersHand = player2Cards->cardSequence();

    m_players[Spades::PLAYER_TWO_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Spades::PLAYER_TWO_INDEX].hand().addCard(playersHand[index]);
}

void SpadesInterface::player3CardMoved()
{
    CardSequence playersHand = player3Cards->cardSequence();

    m_players[Spades::PLAYER_THREE_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Spades::PLAYER_THREE_INDEX].hand().addCard(playersHand[index]);
}

void SpadesInterface::player4CardMoved()
{
    CardSequence playersHand = player4Cards->cardSequence();

    m_players[Spades::PLAYER_FOUR_INDEX].hand().clear();
    for (int index = 0, size = playersHand.size(); index < size; ++index)
        m_players[Spades::PLAYER_FOUR_INDEX].hand().addCard(playersHand[index]);
}

void SpadesInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (event->timerId() == m_computerTimerId)
        if (m_players[Spades::PLAYER_ONE_INDEX].isTurn())
            computerPlay(Spades::PLAYER_ONE_INDEX);
        else if (m_players[Spades::PLAYER_TWO_INDEX].isTurn())
            computerPlay(Spades::PLAYER_TWO_INDEX);
        else if (m_players[Spades::PLAYER_FOUR_INDEX].isTurn())
            computerPlay(Spades::PLAYER_FOUR_INDEX);
        else
            QWidget::timerEvent(event);
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void SpadesInterface::computerPlay(PlayerIndexes playerIndex)
{
    SpadesPlayer *compPlayer=static_cast<SpadesPlayer *>(&m_players[playerIndex]);

    switch (playerIndex)
    {
    case Spades::PLAYER_ONE_INDEX:
        player1Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Spades::PLAYER_TWO_INDEX:
        player2Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Spades::PLAYER_THREE_INDEX:
        throw KardsGTError("SpadesInterface", "computerPlay", "Player three is not a computer player.");
        break;
    case Spades::PLAYER_FOUR_INDEX:
        player4Cards->setSelected(compPlayer->playCard(m_playSequence));
        break;
    case Spades::NON_PLAYER:
        throw KardsGTError("SpadesInterface", "computerPlay", "Not a real player!");
    }
}
