/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef HEARTS_H
#define HEARTS_H

#include "gamebase.h"
#include "heartsrules.h"
#include "carddeck.h"

#include <vector>
using std::vector;

class HeartsInterface;

/**
 * This is the control end for the game of hearts
 * @note Player 3 is our human player.
 *
 * @author John Schneiderman
 */
class Hearts: public GameBase
{
public:
    /**
     * The constructor for the hearts control.
     * @param gameInterface is the interface for the game.
     * @param profileDatabase is the user profile database.
     */
    Hearts(HeartsInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * The destructor.
     */
    ~Hearts();
    /**
     * This starts the game of hearts. The cards are dealt, the cards are passed if needed, and the current player's turn is set.
     * @throw KardsGTError if there are not the right number of players for the game.
     */
    virtual void startPlay();
    /**
     * This saves the game of hearts.
     * @param filename is the name of the file to save the game to.
     * @return true if we successfully save the game, false elsewise.
     */
    virtual bool save(const QString &filename);
    /**
     * This loads the game of hearts.
     * @param filename is the name of the file to load.
     * @return true if we load the game successfully, false elsewise.
     */
    virtual bool load(const QString &filename);
    /**
     * This gives the minimum number of players needed to play the game of hearts.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * This gives the maximum number of players allowed to play the game of hearts.
     * @return
     */
    virtual int maximumPlayers() const;

protected:
    /**
     * These are the indexes for each player.
     * @param PLAYER_ONE_INDEX is the index for player one.
     * @param PLAYER_TWO_INDEX is the index for player two.
     * @param PLAYER_THREE_INDEX is the index for player three.
     * @param PLAYER_FOUR_INDEX is the index for player four.
     * @param NON_PLAYER is the index for an invalid player.
     */
    enum PlayerIndexes { PLAYER_ONE_INDEX=0, PLAYER_TWO_INDEX=1, PLAYER_THREE_INDEX=2, PLAYER_FOUR_INDEX=3, NON_PLAYER=-1 };

    /// @param m_rules is the rules to the game of hearts.
    HeartsRules m_rules;
    /// @param m_cardsAcquired are the cards that each player gets during a phase to count at the end of a round.
    vector<CardSequence> m_cardsAcquired;
    /// @param m_playerCardOrderIndexes is the order of the cards that each player played.
    vector<PlayerIndexes> m_playerCardOrderIndexes;

    /**
     * Handles a card being played. If it's a legal play, then the card is removed from the players hand, added to the play sequence, and the next player's turn is set.
     * @param card is the card the player wants to play.
     * @throw KardsGTError if card cannot be found in the current player's hand.
     */
    virtual void cardPlayed(const Card &card);
    /**
     * Deals out the cards to each player, and then requests them to pass their cards if appropriate.
     */
    virtual void deal();
    /**
     * Disables the current player, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * Determines and handles the various stages in game play.
     * @return the last major GameStatusCode value after handling the changes.
     */
    virtual int handleGameStatus();
    /**
     * This searches for who won the game.
     * @return the index to the player who won the game.
     */
    PlayerIndexes playerWhoWon() const;

private:
    /**
     * These are the available directions to pass cards during the passing phase.
     * @param LEFT pass left for rounds 1, 5, 9...
     * @param RIGHT pass right for rounds 2, 6, 10...
     * @param ACROSS pass across for rounds 3, 7, 11...
     * @param HOLD do not pass for rounds 4, 8, 12...
     */
    enum DirectionToPass { LEFT, RIGHT, ACROSS, HOLD };
    /**
     * These are the game ending states for Hearts.
     * @param NO_CHANGE indicates there has been no change in the current state.
     * @param PHASE_ENDED indicates that the phase has ended.
     * @param ROUND_ENDED indicates that the round has ended.
     * @param GAME_OVER indicates that the game has ended.
     */
    enum GameStatusCodes { NO_CHANGE = 0, PHASE_ENDED = 1, ROUND_ENDED = 2, GAME_OVER = 3 };

    /// @param m_deck is the card deck for the game.
    CardDeck m_deck;
    /// @param m_pInterface is the interface to our graphical interface.
    HeartsInterface *m_pInterface;
    /// @param m_roundNumber is the current round starting with 1 rotating on each 4.
    int m_roundNumber;
    /// @param m_isStart is true if we're at the very start of a round, false elsewise.
    bool m_isStart;
    /// @param m_playerPlayedHighestCard is the index to the player who played the highest card.
    PlayerIndexes m_playerPlayedHighestCard;

    /**
     * Handles the passing of the cards.
     * @throw KardsGTError if a player fails to pass enough cards.
     */
    void passCards();
    /**
     * Determines the direction to pass the cards.
     * @return the direction to pass based on the current round number.
     * @throw KardsGTError if we cannot determine the direction to pass.
     */
    DirectionToPass directionToPass() const;
    /**
     * Actually passes the cards to the appropriate player.
     * @param passedCards are the cards each player has selected.
     * @param direction is the current direction to pass the cards.
     * @throw KardsGTError if a passed card cannot be found in passing player's hand.
     * @throw KardsGTError if the direction to pass is HOLD.
     */
    void passSelectedCards(const vector<CardSequence> &passedCards, Hearts::DirectionToPass direction);
    /**
     * Handles the end of the current phase.
     * @note Doesn't reset the current play sequence.
     */
    void handelEndOfPhase();
    /**
     * This adds the current player to the card playing order.
     */
    void setupCardOrder();
    /**
     * Searches through the playing sequence and finds the highest card that was played.
     * @return the index of the player who played the highest card.
     */
    PlayerIndexes playerPlayedHighestCard() const;
    /**
     * This handles all the tasks to reset the round.
     */
    void handleEndOfRound();
    /**
     * This handles all the scoring for the players.
     */
    void handleScoring();
    /**
     * This handles all the end round messages.
     */
    void handleRoundSummary();
    /**
     * This handles all the tasks for the end of the game.
     */
    void handleGameOver();
    /**
     * Clears the playing field and the cards each of the players had acquired.
     */
    void reset();
};
#endif
