/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "hearts.h"
#include "kardsgterror.h"
#include "heartsinterface.h"
#include "heartsplayer.h"
#include "kardplaysequence.h"

#include <algorithm>
#include <qfile.h>

Hearts::Hearts(HeartsInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_cardsAcquired(), m_playerCardOrderIndexes(), m_deck()
{
    m_pInterface = gameInterface;
    m_roundNumber = 1;
    m_isStart = false;
    m_playerPlayedHighestCard = NON_PLAYER;
}

Hearts::~Hearts()
{}

void Hearts::startPlay()
{
    time_t seconds;
    int rnd;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("Hearts", "startPlay", "Wrong number of players were added to the game.");
    m_players.swap(Hearts::PLAYER_THREE_INDEX, Hearts::PLAYER_FOUR_INDEX); // Because the array structure is intially different than the visual display.
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    rnd=(rand() % 100) + 1;
    if ((rnd > 75) && (rnd <= 100))
        m_dealerIndex=3;
    else if ((rnd > 50) && (rnd <= 75))
        m_dealerIndex=2;
    else if ((rnd > 25) && (rnd <= 50))
        m_dealerIndex=1;
    else
        m_dealerIndex=0;
    deal();
}

bool Hearts::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(IO_WriteOnly | IO_Translate))
        return false;
    out.setEncoding(QTextStream::UnicodeUTF8);
    m_players.swap(Hearts::PLAYER_FOUR_INDEX, Hearts::PLAYER_THREE_INDEX); // Because the human player must be last.

    out << "Game: " << "hearts" << endl;
    // Begin GameBase Save
    out << "Players: " << endl << m_players << endl;
    out << "RoundPoints: " << m_roundPoints << endl;
    out << "DealerIndex: " << m_dealerIndex << endl;
    out << "PlaySequence: " << m_playSequence << endl;
    // End GameBase Save
    out << "Deck: " << m_deck << endl;
    out << "Round: " << m_roundNumber << endl;
    out << "CardOrderIndexesSize: " << m_playerCardOrderIndexes.size() << endl;
    out << "CardOrderIndexes: ";
    for (int i=0; i < static_cast<int>(m_playerCardOrderIndexes.size()); ++i)
        out << m_playerCardOrderIndexes[i] << " ";
    out << endl;
    out << "CardsAquired: ";
    for (int i=0; i < static_cast<int>(m_cardsAcquired.size()); ++i)
        out << m_cardsAcquired[i] << " ";
    out << endl;
    out << "HighestCardPlayerIndex: " << m_playerPlayedHighestCard << endl;
    m_players.swap(Hearts::PLAYER_THREE_INDEX, Hearts::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    if (file.status() != IO_Ok)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool Hearts::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;
    int value, size;

    if (! file.open(IO_ReadOnly | IO_Translate))
    {
        file.close();
        return false;
    }
    in.setEncoding(QTextStream::UnicodeUTF8);
    in >> label >> label;
    if (label != "hearts")
        return false;
    // Begin GameBase Load
    in >> label >> m_players;
    m_players.swap(Hearts::PLAYER_THREE_INDEX, Hearts::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    in >> label >> m_roundPoints;
    in >> label >> m_dealerIndex;
    in >> label >> m_playSequence;
    // End GameBase Load
    in >> label >> m_deck;
    in >> label >> m_roundNumber;
    m_playerCardOrderIndexes.clear();
    in >> label >> size;
    in >> label;
    for (int i=0; i < size; ++i)
    {
        in >> value;
        m_playerCardOrderIndexes.push_back(static_cast<PlayerIndexes>(value));
    }
    m_cardsAcquired.clear();
    in >> label;
    for (int i=0; i < 4; ++i)
    {
        m_cardsAcquired.push_back(CardSequence());
        in >> m_cardsAcquired[i];
    }
    in >> label >> value;
    m_playerPlayedHighestCard=static_cast<PlayerIndexes>(value);
    ;
    file.close();
    m_pInterface->updateTable();
    return true;
}

int Hearts::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int Hearts::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

void Hearts::cardPlayed(const Card &card)
{
    if (m_players.currentPlayer().hand().hasCard(card))
    {
        if (m_rules.isLegalPlay(m_playSequence, card, m_players.currentPlayer()))
        {
            m_playSequence.addCard(card);
            m_players.currentPlayer().hand().removeCard(card);
            setupCardOrder();
            m_pInterface->updateTable();
            GameStatusCodes status = static_cast<GameStatusCodes>(handleGameStatus());
            if ((status != Hearts::ROUND_ENDED) && (status != Hearts::GAME_OVER))
                setupNextPlayer();
        }
    }
    else
        throw KardsGTError(QString("Hearts"), QString("cardPlayed"), QString("Failed to find card %1").arg(card.toString()));
}

void Hearts::deal()
{
    m_pInterface->discardPile->clear(); // We have a delay setup, and so we need to clear the play sequence here.
    for (int i = 0; i < m_players.size(); ++i)
    {
        m_players[i].hand().clear();
        m_cardsAcquired.push_back(CardSequence());
    }
    nextDealer();
    m_deck.shuffle();
    for (int i = 0; i < m_rules.numberOfCardsToDeal(); ++i)
        for (int j = 0; j < m_players.size(); ++j)
            m_players.nextPlayer().hand().addCard(m_deck.dealCard());
    m_isStart = true;
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().sortBySuit(); // Give a usefull arrangement.
    m_pInterface->updateTable(); // So the human player can see who is where.
    passCards();
    setupNextPlayer();
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().sortBySuit(); // Give a usefull arrangement.
    m_pInterface->updateTable();
    m_isStart = false;
}

void Hearts::setupNextPlayer()
{
    if (m_isStart) // Start the turn of the player with the 2c
    {
        Card starterCard(Card::TWO, Card::CLUBS);
        for (int i=0; i < 4; ++i)
            if (m_players[i].hand().hasCard(starterCard))
            {
                m_players.setCurrentPlayer(m_players[i]);
                m_players.currentPlayer().setTurn(true);
                break;
            }
    }
    else if (! m_rules.isGameOver(m_players)) // See if the game is over before we start someones turn.
    {
        m_players.currentPlayer().setTurn(false);
        if (m_rules.isPhaseOver(m_players, m_playSequence))
        {
            m_players.setCurrentPlayer(m_players[m_playerPlayedHighestCard]);
            m_playSequence.clear(); // Clear the play sequence for the next phase. We clear it here because we need the check if the phase is over.
        }
        else
            m_players.nextNonEmptyPlayer();
        m_players.currentPlayer().setTurn(true);
    }
    else
        for (int i=0; i < m_players.size(); ++i)
            m_players[i].setTurn(false);
}

int Hearts::handleGameStatus()
{
    if (m_rules.isPhaseOver(m_players, m_playSequence))
    {
        handelEndOfPhase();
        if (m_rules.isRoundOver(m_players))
        {
            handleEndOfRound();
            if (m_rules.isGameOver(m_players))
            {
                handleGameOver();
                return Hearts::GAME_OVER;
            }
            else
                return Hearts::ROUND_ENDED;
        }
        else
            return Hearts::PHASE_ENDED;
    }
    return Hearts::NO_CHANGE;
}

void Hearts::passCards()
{
    vector<CardSequence> passedCards;
    DirectionToPass passDirection;
    QString directionName;

    // Determine the direction to pass
    passDirection=directionToPass();
    switch (passDirection)
    {
    case Hearts::LEFT:
        directionName="left";
        break;
    case Hearts::RIGHT:
        directionName="right";
        break;
    case Hearts::ACROSS:
        directionName="across";
        break;
    case Hearts::HOLD:
        return;
    };
    // Let players select their cards.
    for (int i=0; i < m_players.size(); ++i)
    {
        HeartsPlayer *heartsPlayer=dynamic_cast<HeartsPlayer *>(&m_players[i]);

        if (heartsPlayer == 0)
            passedCards.push_back(m_pInterface->passCards(m_players[i], directionName));
        else
            passedCards.push_back(heartsPlayer->passCards(m_playSequence));
        if (passedCards[i].size() != HeartsRules::PASS_TO_PLAYER)
            throw KardsGTError(QString("hearts"), QString("passCards"), QString("Player %1 failed to pass enough cards!").arg(m_players[i].name()));
    }
    // Pass the cards to the appropriate players.
    passSelectedCards(passedCards, passDirection);
}

Hearts::DirectionToPass Hearts::directionToPass() const
{
    switch (m_roundNumber)
    {
    case 4:
        return Hearts::HOLD;
    case 3:
        return Hearts::ACROSS;
    case 2:
        return Hearts::RIGHT;
    case 1:
        return Hearts::LEFT;
    default:
        throw KardsGTError("Hearts", "directionToPass", "Failed to determine direction to pass");
    }
}

void Hearts::passSelectedCards(const vector<CardSequence> &passedCards, Hearts::DirectionToPass direction)
{
    // Remove selected cards from the players hands.
    for (int player=0; player < m_players.size(); ++player)
        for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
            if (! m_players[player].hand().removeCard(passedCards[player][cardIndex]))
                throw KardsGTError("Hearts", "passSelectedCards", "Failed to find a players card!");
    // Pass the selected cards to the respective players.
    for (int player=0; player < m_players.size(); ++player)
    {
        /* Player Arrangement based on the interface and their location in the passedCards array.
                                Player 0  (One)
            Player 3 (Four)                            Player 1 (Two)
                             Player 2 (human- Three)
        */
        switch (direction)
        {
        case Hearts::LEFT:
            if (player == 0) // Player 1
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_TWO_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 1) // Player 2
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_THREE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 2) // Player 3
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_FOUR_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 3) // Player 4
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_ONE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            break;
        case Hearts::RIGHT:
            if (player == 0) // Player 1
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_FOUR_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 1) // Player 2
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_ONE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 2) // Player 3
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_TWO_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 3) // Player 4
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_THREE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            break;
        case Hearts::ACROSS:
            if (player == 0) // Player 1
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_THREE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 1) // Player 2
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_FOUR_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 2) // Player 3
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_ONE_INDEX].hand().addCard(passedCards[player][cardIndex]);
            else if (player == 3) // Player 4
                for (int cardIndex=0; cardIndex < passedCards[player].size(); ++cardIndex)
                    m_players[Hearts::PLAYER_TWO_INDEX].hand().addCard(passedCards[player][cardIndex]);
            break;
        case Hearts::HOLD:
            // We should never get this case.
            throw KardsGTError("Hearts", "passSelectedCards", "Hold condition was met.");
        }
    }
}

void Hearts::handelEndOfPhase()
{
    CardSequence counters;

    // Keep track of the cards each player gets for the phase.
    m_playerPlayedHighestCard=playerPlayedHighestCard();
    counters = m_rules.cardPointers(m_playSequence);
    for (int i=0; i < counters.size(); ++i)
        m_cardsAcquired[m_playerPlayedHighestCard].addCard(counters[i]);
    m_playerCardOrderIndexes.clear(); // We're going to start a new phase, clear our card order tracker.
}

void Hearts::setupCardOrder()
{
    if (m_players.currentPlayer() == m_players[Hearts::PLAYER_ONE_INDEX])
        m_playerCardOrderIndexes.push_back(Hearts::PLAYER_ONE_INDEX);
    else if (m_players.currentPlayer() == m_players[Hearts::PLAYER_TWO_INDEX])
        m_playerCardOrderIndexes.push_back(Hearts::PLAYER_TWO_INDEX);
    else if (m_players.currentPlayer() == m_players[Hearts::PLAYER_THREE_INDEX])
        m_playerCardOrderIndexes.push_back(Hearts::PLAYER_THREE_INDEX);
    else if (m_players.currentPlayer() == m_players[Hearts::PLAYER_FOUR_INDEX])
        m_playerCardOrderIndexes.push_back(Hearts::PLAYER_FOUR_INDEX);
    else
        m_playerCardOrderIndexes.push_back(Hearts::NON_PLAYER);
}

Hearts::PlayerIndexes Hearts::playerPlayedHighestCard() const
{
    int highCardFound=0;
    Card::suit_t leadingSuit;

    leadingSuit = m_playSequence.front().suit();
    for (int cardIndex=0; cardIndex < m_playSequence.size(); ++cardIndex)
        if (m_playSequence[cardIndex].suit() == leadingSuit)
            if (m_rules.rankValue(m_playSequence[cardIndex]) > m_rules.rankValue(m_playSequence[highCardFound]))
                highCardFound = cardIndex;
    return m_playerCardOrderIndexes[highCardFound];
}

void Hearts::handleEndOfRound()
{
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].setTurn(false);
    m_pInterface->updateTable(); // So that the table doesn't look like the last player hasn't played his card.
    handleScoring(); // Take care of the players score's.
    handleRoundSummary(); // Creates the final round summay information and show it to the user.
    m_playerPlayedHighestCard = NON_PLAYER;
    m_roundNumber = ++m_roundNumber % 4; // Keep the round numbers between 1 and 4.
    if (m_roundNumber == 0)
        m_roundNumber = 4;
    reset();
    if (! m_rules.isGameOver(m_players))
        deal();
}

void Hearts::handleScoring()
{
    bool shotTheMoon=false;
    PlayerIndexes playerWhoShot=NON_PLAYER;

    // Check to see if any player has shot the moon
    for (int playerIndex=0; playerIndex < m_players.size(); ++playerIndex)
        if (m_rules.hasShotTheMoon(m_cardsAcquired[playerIndex]))
        {
            shotTheMoon = true;
            playerWhoShot = static_cast<PlayerIndexes>(playerIndex);
            break;
        }
    // If a player has shot the moon add penalty scores to the other players, else add the points for the other players.
    if (shotTheMoon)
    {
        for (int playerIndex=0; playerIndex < m_players.size(); ++playerIndex)
            if (playerIndex != playerWhoShot)
                m_players[playerIndex].addPoint(HeartsRules::SHOT_THE_MOON);
    }
    else
        for (int playerIndex=0; playerIndex < m_players.size(); ++playerIndex)
            for (int cardIndex=0; cardIndex < m_cardsAcquired[playerIndex].size(); ++cardIndex)
                m_players[playerIndex].addPoint(m_rules.cardValue(m_cardsAcquired[playerIndex][cardIndex]));
}

void Hearts::handleRoundSummary()
{
    vector<CardSequence> cards;
    vector<QString> messages;

    for (int i=0; i < m_players.size(); ++i)
    {
        cards.push_back(m_cardsAcquired[i]);
        messages.push_back(QString("%1 has a score of %2.").arg(m_players[i].name()).arg(m_players[i].score()));
    }
    m_pInterface->displayRoundSummary("Scores at the end of the round", cards, messages);
}

void Hearts::handleGameOver()
{
    m_pInterface->gameWon();
}

void Hearts::reset()
{
    m_cardsAcquired.clear();
    m_playSequence.clear();
}

Hearts::PlayerIndexes Hearts::playerWhoWon() const
{
    PlayerIndexes player=Hearts::NON_PLAYER;
    int lowestScore=m_players[PLAYER_ONE_INDEX].score() + 1; // Make the score initially not the lowest.

    for (int i=0; i < m_players.size(); ++i)
        if (m_players[i].score() < lowestScore)
        {
            player = static_cast<PlayerIndexes>(i);
            lowestScore = m_players[i].score();
        }
    return player;
}
