/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "userprofiledatabase.h"
#include "kardsgterror.h"

#include <qfile.h>
#include <qtextstream.h>

// Constants used for portability
const QString UserProfileDatabase::INVALID_USER="~~~INVALID_USER~~~";
const QString UserProfileDatabase::MAKE_DIRECTORY="mkdir";
const QString UserProfileDatabase::MAKE_DIRECTORY_PATH="mkdir -p";
const QString UserProfileDatabase::CHANGE_DIRECTORY="cd";
const QString UserProfileDatabase::REMOVE_DIRECTORY="rm -fdr";
const QString UserProfileDatabase::DIRECTORY_MARKER="/";
const QString UserProfileDatabase::GAME_DIRECTORY=".kardsgt";
const QString UserProfileDatabase::PROFILE_DIRECTORY=UserProfileDatabase::GAME_DIRECTORY + UserProfileDatabase::DIRECTORY_MARKER + "profiles";
const QString UserProfileDatabase::STATISTICS_EXTENSION=".stat";
const QString UserProfileDatabase::PROFILE_DATABASE_NAME="profiles.lst";
const QString UserProfileDatabase::USER_SETTINGS_EXTENSION=".kus";

UserProfileDatabase::UserProfileDatabase(const QString &path): m_users()
{
    if (path.isEmpty())
        throw KardsGTError("UserProfileDatabase", "UserProfileDatabase", "path is empty!");
    m_basePath=path;
}

UserProfileDatabase::~UserProfileDatabase()
{}

vector<QString> UserProfileDatabase::users() const
{
    return m_users;
}

void UserProfileDatabase::setUsers(const vector<QString> &users)
{
    if (users.size() > m_users.size()) // Add a user profile
        addUser(users);
    else if (users.size() < m_users.size()) // remove a user profile
        removeUser(users);
}

pair<int, int> UserProfileDatabase::userStatistics(const QString &user, const QString &game) const
{
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + game + STATISTICS_EXTENSION);
    QTextStream in(&file);
    pair<int, int> stat;
    QString label;

    stat.first = 0;
    stat.second = 0;
    if (file.open(IO_ReadOnly | IO_Translate))
    {
        in.setEncoding(QTextStream::UnicodeUTF8);
        in >> label >> stat.first;
        in >> label >> stat.second;
    }
    file.close();
    return stat;
}

void UserProfileDatabase::setUserGamesWon(const QString &user, const QString &game, bool won)
{
    pair<int, int> stat;
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + game + STATISTICS_EXTENSION);
    QTextStream out(&file);

    stat=userStatistics(user, game);
    if (won)
        stat.first++;
    stat.second++;
    if (! file.open(IO_WriteOnly | IO_Translate))
    {
        file.close();
        if (! file.open(IO_WriteOnly | IO_Translate)) // Attempt to create the database
            throw KardsGTError(QString("UserProfileDatabase"), QString("setUserGamesWon"), QString("Could not create game profile: %1").arg(file.name()));
        out << QString("Won: %1").arg(stat.first) << endl;
        out << QString("Played: %1").arg(stat.second) << endl;
    }
    else
    {
        out << QString("Won: %1").arg(stat.first) << endl;
        out << QString("Played: %1").arg(stat.second) << endl;
    }
    file.close();
}

void UserProfileDatabase::loadUserProfileDatabase()
{
    if (system(CHANGE_DIRECTORY + " " + m_basePath + DIRECTORY_MARKER + GAME_DIRECTORY) != 0)
        system(MAKE_DIRECTORY_PATH + " " + m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY);
    QFile file(m_basePath + DIRECTORY_MARKER + GAME_DIRECTORY + DIRECTORY_MARKER + PROFILE_DATABASE_NAME);
    QTextStream in(&file);
    QString user;

    if (! file.open(IO_ReadOnly | IO_Translate))
    {
        file.close();
        // Create the file
        if (! file.open(IO_WriteOnly | IO_Translate))
            throw KardsGTError(QString("UserProfileDatabase"), QString("loadUserProfileDatabases"), QString("Could not create profile database: %1").arg(file.name()));
        file.close();
    }
    else
    {
        in.setEncoding(QTextStream::UnicodeUTF8);
        while (! in.atEnd())
        {
            in >> user;
            m_users.push_back(user);
        }
        if (file.status() != IO_Ok)
        {
            file.close();
            throw KardsGTError(QString("UserProfileDatabase"), QString("loadUserProfileDatabases"), QString("Could not read profile database: %1").arg(file.name()));
        }
        file.close();
    }
}

void UserProfileDatabase::saveUserProfileDatabase()
{
    QFile file(m_basePath + DIRECTORY_MARKER + GAME_DIRECTORY + DIRECTORY_MARKER + PROFILE_DATABASE_NAME);
    QTextStream out(&file);

    if (! file.open(IO_WriteOnly | IO_Translate))
        throw KardsGTError(QString("UserProfileDatabase"), QString("saveUserProfileDatabases"), QString("Could not save the profile database: %1").arg(file.name()));
    else
        for (int i=0, size=m_users.size(); i < size; ++i)
            out << m_users[i] << endl;
    file.close();
}

void UserProfileDatabase::addUser(const vector<QString> &users)
{
    QString user = INVALID_USER;
    bool userFound=false;

    // Look for the new user.
    for (int i=0, usersSize=users.size(); i < usersSize; ++i)
    {
        userFound=false;
        for (int j=0, size=m_users.size(); j < size; ++j)
            if (users[i] == m_users[j]) // Are we looking at the new user?
            {
                userFound=true;
                break;
            }
        if (! userFound) // If we couldn't find the user in our current DB, then we're looking at the new user.
        {
            user=users[i];
            break;
        }
    }
    if (user != INVALID_USER)
        system(MAKE_DIRECTORY + " " + m_basePath + DIRECTORY_MARKER +  PROFILE_DIRECTORY + DIRECTORY_MARKER + user);
    m_users.clear();
    m_users=users;
    saveUserProfileDatabase(); // This will prevent the profile directories from being different than our list.
}

void UserProfileDatabase::removeUser(const vector<QString> &users)
{
    QString user=INVALID_USER;
    bool userFound=false;

    // Search through all the users.
    for (int i=0, size=m_users.size(); i < size; ++i)
    {
        userFound=false;
        for (int j=0, usersSize=users.size(); j < usersSize; ++j)
            if (m_users[i] == users[j]) // Are we looking at the missing user?
            {
                userFound=true;
                break;
            }
        if (! userFound) // If we couldn't find the user in our new DB, than we've found the old user.
        {
            user=m_users[i];
            break;
        }
    }
    if (user != INVALID_USER)
        system(REMOVE_DIRECTORY + " " + m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user);
    m_users.clear();
    m_users=users;
    saveUserProfileDatabase(); // This will prevent the profile directories from being different than our list.
}

QString UserProfileDatabase::userImage(const QString &user, const QString &mood) const
{
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + user + USER_SETTINGS_EXTENSION);
    QTextStream in(&file);
    QString moodLabel, imageName = "player-male.png"; // Default player image in mime source.

    if (file.open(IO_ReadOnly | IO_Translate))
        while (! in.atEnd())
        {
            in.setEncoding(QTextStream::UnicodeUTF8);
            in >> moodLabel >> imageName;
            if (moodLabel == mood)
                break;
        }
    file.close();
    return imageName.replace("~", " ");
}

QString UserProfileDatabase::pathToCardImages(const QString &user) const
{
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + user + USER_SETTINGS_EXTENSION);
    QTextStream in(&file);
    QString label, cardPath = "default"; // Default is from mime source

    if (file.open(IO_ReadOnly | IO_Translate))
        while (! in.atEnd())
        {
            in.setEncoding(QTextStream::UnicodeUTF8);
            in >> label >> cardPath;
            if (label == "CardImagePath")
                break;
        }
    file.close();
    return cardPath.replace("~", " ");
}

QString UserProfileDatabase::pathToCardBackImage(const QString &user) const
{
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + user + USER_SETTINGS_EXTENSION);
    QTextStream in(&file);
    QString label, cardBackImage = "cardback.png"; // Default is in mime source as cardBack.png

    if (file.open(IO_ReadOnly | IO_Translate))
        while (! in.atEnd())
        {
            in.setEncoding(QTextStream::UnicodeUTF8);
            in >> label >> cardBackImage;
            if (label == "CardBackPath")
                break;
        }
    file.close();
    return cardBackImage.replace("~", " ");
}

void UserProfileDatabase::setUserSettings(const QString &user, const vector<pair<QString, QString> > &settings)
{
    QFile file(m_basePath + DIRECTORY_MARKER + PROFILE_DIRECTORY + DIRECTORY_MARKER + user + DIRECTORY_MARKER + user + USER_SETTINGS_EXTENSION);
    QTextStream out(&file);
    vector<pair<QString, QString> > userSettings = settings;

    if (! file.open(IO_WriteOnly | IO_Translate))
        throw KardsGTError(QString("UserProfileDatabase"), QString("setUserSettings"), QString("Could not save the user's settings: %1").arg(file.name()));
    else
        for (int i=0, size=settings.size(); i < size; ++i)
            out << userSettings[i].first << " " << userSettings[i].second.replace(" ", "~") << endl;
    file.close();
}
