/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef GAMEBASE_H
#define GAMEBASE_H

#include "playerlist.h"
#include "cardsequence.h"

class ComputerPlayer;
class Player;
class QString;
class UserProfileDatabase;
class RuleBase;

/**
 * This is the base interface that all the card games must derive from.
 * This class is used in connection with the @ref RuleBase class.
 * This class should contain the logical flow and enforce the rules of the game.
 * If a method is not needed for a specific game, the method should throw a KardsGTError to prevent it's usage.
 *
 * @author John Schneiderman
 */
class GameBase
{
public:
    /**
     * Default constructor.
     * @param profileDatabase is the database of the user profiles.
     */
    GameBase(UserProfileDatabase &profileDatabase);
    /**
     * Destructor.
     */
    virtual ~GameBase();
    /**
     * This method when implemented should begin all the necessary steps to start the game.
     */
    virtual void startPlay()=0;
    /**
     * This will save the game when implemented. \n
     * The following format is expected: \n
     * label: value(s) \n
     * Each label-values should be on its own line. \n
     * The first label should be the name of the game, followed by each players information. \n
     * The last player in the player listings must be the human player.
     * @param filename is the name of the file to save the game to.
     * @return true if the game was successfully saved, false elsewise.
     */
    virtual bool save(const QString &filename)=0;
    /**
     * This will load the game when implemented.
     * @param filename is the name of the file to load the game from.
     * @return true if the game was successfully loaded, false elsewise.
     */
    virtual bool load(const QString &filename)=0;
    /**
     * This will give the minimum number of players needed to play the game.
     * @note The actual value should come from a @ref RuleBase call.
     * @return the number of players needed.
     */
    virtual int minimumPlayers() const=0;
    /**
     * This will give the maximum number of players allowed to play the game.
     * @note The actual value should come from a @ref RuleBase call.
     * @return the number of players allowed.
     */
    virtual int maximumPlayers() const=0;
    /**
     * This will handle the game status changes.
     * @return a code to indicate the type of status change. All codes are game specific.
     */
    virtual int handleGameStatus()=0;
    /**
     * Adds the opponent to play against the player.
     * @param opponent is the player to play against our local player.
     */
    virtual void addOpponent(ComputerPlayer &opponent);
    /**
     * Adds the local player.
     * @param player is the local player.
     */
    void addPlayer(Player &player);
    /**
     * Resets all values to be empty.
     * @param roundOver is true if the round is over, false elsewise.
     * @note When roundOver is true the player's round score is reset. See @ref Player class for more information.
     */
    void resetGame(bool roundOver);

protected:
    /// @param m_players is all the players in the game.
    PlayerList m_players;
    /// @param m_roundPoints is the total number of points in the round of play.
    int m_roundPoints;
    /// @param m_dealerIndex is the index in the player's list of the dealer.
    int m_dealerIndex;
    /// @param m_playSequence is the current card play sequence in a round or phase.
    CardSequence m_playSequence;
    /// @param m_pProfileDatabase is the pointer to the current profile database.
    UserProfileDatabase *m_pProfileDatabase;

    /**
     * This is the method to call when a card has been selected to play when implemented.
     * @param card is the card that the player has selected.
     */
    virtual void cardPlayed(const Card &card)=0;
    /**
     * Deals the cards to the players when implemented.
     */
    virtual void deal()=0;
    /**
     * This sets up the turn for next player in the player's list.
     */
    virtual void setupNextPlayer()=0;
    /**
     * Sets the dealer index to the next dealer.
     * @note Dealer is rotated in a clockwise motion.
     */
    void nextDealer();
};
#endif
