/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef NORMANHEARTSAI_H
#define NORMANHEARTSAI_H

#include "cardsequence.h"
#include "rulebase.h"

/**
 * This is how Norman plays hearts.
 *
 * @author John Schneiderman
 */
class NormanHeartsAI
{
public:
    /**
     * The general constructor.
     * @param playSequence is the current play sequence.
     * @param rules are the rules for the game.
     * @param hand is Norman's hand.
     */
    NormanHeartsAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand);
    /**
     * The general destructor.
     */
    ~NormanHeartsAI();
    /**
     * Selects a card to play.
     * @return the card selected.
     */
    CardSequence selectCards() const;
    /**
     * Selects cards to pass.
     * @return the cards selected.
     */
    CardSequence passCards() const;

private:
    /// @param m_playSequence is the play sequence we're working with.
    CardSequence m_playSequence;
    /// @param m_hand is the hand of cards we're working with.
    CardSequence m_hand;
    /// @param m_pRules are the rules for the game.
    const RuleBase *m_pRules;

    /**
     * Determines if a sequence of cards are guarded. \n
     * A guarded card has 3 or more low cards (less than 8).
     * @param sequence is the sequence of cards to examine.
     * @return true if the sequence is guarded, false elsewise.
     */
    bool isGuarded(const CardSequence &sequence) const;
    /**
     * Combines the high cards out of a sequence of cards.
     * @param sequence is the sequence of cards to put the extracted high cards into.
     * @param toGuard is the sequence of cards to extract the high cards from.
     */
    void addUnguardedHighCards(CardSequence &sequence, const CardSequence &toGuard) const;
    /**
     * This removes some of the cards from a sequence.
     * @param sequence is the sequence of cards to put the shaved cards into.
     * @param shaveSuit is the sequence of cards to remove cards from.
     */
    void shaveLargeSuits(CardSequence &sequence, const CardSequence &shaveSuit) const;
    /**
     * Finds the lowest card in a sequence.
     * @param sequence is the sequence of cards to search through.
     * @return the lowest card found.
     */
    Card lowestCard(const CardSequence &sequence) const;
    /**
     * Find the highest card in a sequence.
     * @param sequence is the sequence of cards we want to search through.
     * @return the highest card found.
     * @throw KardsGTError if we can't find a valid high card.
     */
    Card highestCard(const CardSequence &sequence) const;
    /**
     * Finds the highest card in the hand.
     * @return the card that is the first highest in the hand.
     */
    Card findHighestCardToPlay() const;
};
#endif
