/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef WARINTERFACE_H
#define WARINTERFACE_H

#include "warinterfacebase.h"
#include "war.h"

class UserProfileDatabase;
class Kard;

/**
 * This is the interface to play a game of War.
 *
 * @author John Schneiderman
 */
class WarInterface: public WarInterfaceBase, public War
{
    Q_OBJECT

public:
    /**
     * Default constructor.
     * @param profileDatabase is the user profile database for the person playing the game.
     * @param parent is the parent window.
     * @param name is the name of our widget.
     */
    WarInterface(UserProfileDatabase &profileDatabase, QWidget *parent = 0, const char *name = 0);
    /**
     * Default destructor.
     */
    ~WarInterface();
    /**
     * Updates all of the widgets.
     */
    void updateTable();
    /**
     * Displays the information when a player has won the game.
     * @throw KardsGTError if we cannot determine who won the game.
     */
    void gameWon();
    /**
     * Displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * Prompts a message.
     * @param caption is caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * Shows the cards won after a war.
     * @param player is the player who won the cards.
     * @param cards are the cards the player has won in the war.
     */
    void showWarCards(const Player &player,  const CardSequence &cards);

protected slots:
    /**
     * Handles the kard selected by player 1.
     * @param kard is the card the player selected.
     */
    void player1CardPlayed(Kard &kard);
    /**
     * Handles the kard selected by player 2.
     * @param kard is the card the player selected.
     */
    void player2CardPlayed(Kard &kard);

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence. @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths { COMPUTER_PLAYER_TIME=1500, CLEARING_DELAY_TIME=500 };
    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;

    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
    /**
     * This currently handles the computer playing mechanism.
     * @throw KardsGTError if we cannot find the card the player selected.
     */
    void computerPlay();
};
#endif
