/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CRIBBAGE_H
#define CRIBBAGE_H

#include "carddeck.h"
#include "cribbagerules.h"
#include "gamebase.h"
#include "player.h"

#include <vector>
using std::vector;
using std::iterator;

class ComputerPlayer;
class CribbageInterface;
class CardSequence;
class QString;
class CardProperties;
class UserProfileDatabase;

/**
 * This controls the game of cribbage.
 * @note We currently always play seven stakes for the game.
 *
 * @author John Schneiderman
 */
class Cribbage:public GameBase
{
public:
    /**
     * Default constructor.
     * @param gameInterface is the interface to the game.
     * @param profileDatabase is the user profile database.
     */
    Cribbage(CribbageInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * Default destructor.
     */
    ~Cribbage();
    /**
     * Starts the game of cribbage.
     * @throw KardsGTError if the number of players is incorrect.
     */
    virtual void startPlay();
    /**
     * This will save a game of cribbage.
     * @param filename is the name of the saved file.
     * @return true if the game was saved successfully.
     */
    virtual bool save(const QString &filename);
    /**
     * This will load a game of cribbage.
     * @param filename is the name of the file to load.
     * @return true if the game was loaded successfully.
     */
    virtual bool load(const QString &filename);
    /**
     * This is the minimum number of players needed for a game of Cribbage.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * This is the maximum number of players allowed for a game of Cribbage.
     * @return the maximum number of players allowed.
     */
    virtual int maximumPlayers() const;

protected:
    /// @param m_starter is the starter card.
    Card m_starter;
    /// @param m_crib are the cards passed to the crib.
    CardSequence m_crib;
    /// @param m_rules is the rules on how to play cribbage.
    CribbageRules m_rules;
    /// @param m_deck is the deck of cards that the game plays with.
    CardDeck m_deck;

    /**
     * Deals the cards to the players.
     */
    virtual void deal();
    /**
     * Calculates the scores for the players at the end of each round.
     */
    void score();
    /**
     * Requests that a player discards to the crib.
     * @param player is the player to discard to the crib.
     */
    void discardToCrib(Player &player);
    /**
     * This is the method to call when a card has been selected to play.
     * @param card is the card that the player has selected.
     * @throw KardsGTError if the card is not in the players hand.
     */
    virtual void cardPlayed(const Card &card);
    /**
     * This calculates the score during a phase.
     * @param player is the player who has just played a card.
     */
    void scoreInPlay(Player &player);
    /**
     * Checks to see if a player is in go.
     * @param player is the player to check.
     * @return true if a player is in go, false elsewise.
     */
    bool haveGO(const Player &player);
    /**
     * This advances the next player, if appropriate.
     * Also checks to see if the round is over.
     * @return not used.
     */
    virtual int handleGameStatus();
    /**
     * Disables the current player's turn, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * This checks to see if the current player is in go.
     */
    void checkGo();
    /**
     * This handles the necessary stuff for when the round is over.
     */
    void roundOver();
    /**
     * This handles the necessary stuff for when the game is over.
     */
    void gameOver();
    /**
     * This resets the game either for a phase or for a round.
     * @param roundOver if true will reset the player's score as well.
     */
    void resetGame(bool roundOver);
    /**
     * Does the scoring for a player.
     * @param player is the player to calculate the score for.
     * @param cardProperties is the properties the player's cards contain.
     */
    void scoring(Player &player, const CardProperties &cardProperties);
    /**
     * Finds the combinations of 15 for a player.
     * @param cardProperties is the properties the player's cards contain.
     * @return the points earned for 15.
     */
    int scoreFifteen(const CardProperties &cardProperties);
    /**
     * Finds the pairs a player has.
     * @param cardProperties is the properties the player's cards contain.
     * @return the points earned for each pair.
     */
    int scorePairs(const CardProperties &cardProperties);
    /**
     * Finds the runs a player has.
     * @param cardProperties is the properties the player's cards contain.
     * @return the points earned for each run.
     */
    int scoreRuns(const CardProperties &cardProperties);
    /**
     * Finds the flushes in a player has.
     * @param cardProperties is the properties the player's cards contain.
     * @return the points earned for each flush.
     */
    int scoreFlush(const CardProperties &cardProperties);
    /**
     * Finds the nobs in a players hand.
     * @param player is the player whose had is to be examine.
     * @return the points earned for nobs.
     */
    int scoreNobs(const Player &player);
    /**
     * This removes the pair found in a run sequence from the message queue.
     * @param run is the special run.
     * @throw KardsGTError if the pair type is not able to be determined.
     * @return the number of points scored from the pair.
     */
    int removePairPoints(const CardSequence &run);
    /**
     * Checks to see if any player has won the round.
     * @return true if a player has won the round, false elsewise.
     */
    bool hasAPlayerWonTheRound() const;

private:
    /// @param m_goPlayer is the player who is currently in Go.
    Player m_goPlayer;
    /// @param m_inGo tells whether the game is in a Go situation.
    bool m_inGo;
    /// @param m_scoringCrib tells whether the game is currently scoring the crib or not.
    bool m_scoringCrib;
    /// @param m_savedHands are the hands of all the players at the beginning of the round.
    vector<CardSequence> m_savedHands;
    /// @param m_cardSummary is the summary of the combinations of cards that a players has.
    vector<CardSequence> m_cardSummary;
    /// @param m_messageSummary is the summary of the messages for the cards.
    vector<QString> m_messageSummary;
    /// @param m_pInterface is the pointer to the graphical interface of the game.
    CribbageInterface *m_pInterface;

    /**
     * Saves all the players hands.
     */
    void saveHands();
    /**
     * Restores the players hands that were saved.
     * @note saveHands() needs to have been called before restore can be called.
     */
    void restoreHands();
};
#endif
