/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cardproperties.h"
#include "rulebase.h"
#include "bitmanipulator.h"

CardProperties::CardProperties():m_cardSequences()
{}

CardProperties::CardProperties(const CardSequence &sequence)
{
    setSequence(sequence);
}

const CardSequence& CardProperties::sequence() const
{
    // The last sequence is always every card in our card sequence.
    return m_cardSequences.back();
}

void CardProperties::setSequence(const CardSequence &sequence)
{
    m_cardSequences = parseCardSequence(sequence);
}

vector<Card> CardProperties::singletons() const
{
    vector<Card> singletons;

    return singletons;
}

vector<CardSequence> CardProperties::doubletons() const
{
    vector<CardSequence> doubletons;

    return doubletons;
}

vector<CardSequence> CardProperties::pairs() const
{
    vector<CardSequence> pairs;
    int size=m_cardSequences.size();

    for (int i = 0; i < size; i++) // Loop through all the card combinations
    {
        if (m_cardSequences[i].size() != PAIR) // Can it meet the definition of a pair.
            continue;
        if (m_cardSequences[i][0].rank() == m_cardSequences[i][1].rank())
            pairs.push_back(m_cardSequences[i]);
    }
    return pairs;
}

vector<CardSequence> CardProperties::runs() const
{
    vector<CardSequence> runs;
    bool runNotFound=false;
    int sizeLower=0, size=m_cardSequences.size();

    for (int i = 0; i < size; i++) // Loop through all the card combinations
    {
        if (m_cardSequences[i].size() < RUN) // Can it meet the definition of a run.
            continue;
        runNotFound = false;
        sizeLower = m_cardSequences[i].size() - 1; // Lower our loop because of the comparison.
        for (int j = 0; j < sizeLower; j++)
            if ((m_cardSequences[i][j].rank() - 1) != m_cardSequences[i][j + 1].rank()) // If there is not a difference of 1, not a run.
            {
                runNotFound = true;
                break;
            }
        if (! runNotFound)
            runs.push_back(m_cardSequences[i]);
    }
    return uniqueSequences(runs);
}

vector<CardSequence> CardProperties::flushes() const
{
    vector<CardSequence> flushes;
    bool flushNotFound=false;
    int sizeLower=0, size=m_cardSequences.size();

    for (int i = 0; i < size; i++) // Loop through all the card combinations
    {
        if (m_cardSequences[i].size() < FLUSH) // Can it meet the definition of a flush.
            continue;
        flushNotFound = false;
        sizeLower = m_cardSequences[i].size() - 1;
        for (int j = 0; j <  sizeLower; j++) // Lower our loop because of the comparison.
            if (m_cardSequences[i][j].suit() != m_cardSequences[i][j + 1].suit())
            {
                flushNotFound = true;
                break;
            }
        if (! flushNotFound)
            flushes.push_back(m_cardSequences[i]);
    }
    return uniqueSequences(flushes);
}

vector<CardSequence> CardProperties::summations(int goal, const RuleBase &rules) const
{
    vector<CardSequence> sums;
    int sum, size=m_cardSequences.size();

    for (int i = 0; i < size; i++) // Examine all sequences
    {
        sum = 0;
        for (int j = 0; j < m_cardSequences[i].size(); j++) // Sum up the sequence
            sum = sum + rules.cardValue(m_cardSequences[i][j]);
        if (sum == goal)
            sums.push_back(m_cardSequences[i]);
    }
    return sums;
}

CardSequence CardProperties::suits(Card::suit_t suit) const
{
    CardSequence suits;
    CardSequence allCards=m_cardSequences.back();
    int size = allCards.size();

    for (int index=0; index < size; ++ index)
        if (allCards[index].suit() == suit)
            suits.addCard(allCards[index]);
    return suits;
}

CardSequence CardProperties::ranks(Card::rank_t rank) const
{
    CardSequence ranks;
    CardSequence allCards=m_cardSequences.back();
    int size = allCards.size();

    for (int index=0; index < size; ++ index)
        if (allCards[index].rank() == rank)
            ranks.addCard(allCards[index]);
    return ranks;
}

vector<CardSequence> CardProperties::parseCardSequence(const CardSequence &seq) const
{
    //Dr. Tim Sorenson's idea, Augustana College, Sioux Falls, SD, USA.
    vector<string> binary;
    vector<CardSequence> seqs;
    CardSequence sequence(seq);

    sort(sequence.begin(), sequence.end()); // Sort the sequence
    reverse(sequence.begin(), sequence.end()); // Base2 is read right to left, we'll do the same.
    binary = BitManipulator::binarySequence(sequence.size()); // Get all the base 2 numbers for the length of our sequence.
    for (unsigned i = 0; i < binary.size(); i++)
    {
        seqs.push_back(CardSequence()); // Create our first sequence to work with.
        for (unsigned j = 0; j < binary[i].length(); j++)
            if (binary[i][j] == '1') // If it's a one add the card in that bit position.
                seqs[i].addCard(sequence[j]);
    }
    return seqs;
}

vector<CardSequence> CardProperties::uniqueSequences(const vector<CardSequence> &examineSequence) const
{
    vector<CardSequence> uniqueSeq, examineSeq=examineSequence;
    int size=examineSeq.size(), examineSize=0;
    CardSequence testSeq;

    if (examineSeq.size() > 0)
    {
        // Reverse order so that the smallest card is the first card in all the sequences.
        for (int i=0; i < size; ++i)
            reverse(examineSeq[i].begin(), examineSeq[i].end());
        // Find Unique sequences
        for (int i=0; i < size; ++i)
        {
            for (int j=0; j < size; ++j)
            {
                testSeq=examineSeq[i];
                examineSize=examineSeq[j].size();
                if (i == j) // Don't want to compare the same sequence.
                    continue;
                for (int z=0; z < examineSize; ++z) // Does the test sequence have the same card in our current examining sequence.
                    if (testSeq.hasCard(examineSeq[j][z]))
                        testSeq.removeCard(examineSeq[j][z]);
                if (testSeq.isEmpty())
                    break;
            }
            if (! testSeq.isEmpty()) // If we have remaining unique cards, then we have a unique sequence.
                uniqueSeq.push_back(examineSeq[i]);
        }
    }
    return uniqueSeq;
}
