/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CARDPILE_H
#define CARDPILE_H

#include "card.h"
#include "cardsequence.h"

#include <iostream>
using std::iostream;

class QTextStream;

/**
 * This represents a pile of cards. This is usually used as a discard area.
 *
 * @author John Schneiderman
 */
class CardPile
{
public:
    /**
     * This is the default constructor.
     */
    CardPile();
    /**
     * This is the copy constructor for a @ref CardSequence .
     * @param sequence is the of cards to put into a pile.
     * @note that the first card in the sequence is the bottom card in the pile.
     */
    CardPile(const CardSequence &sequence);
    /**
     * This will clear the card pile.
     */
    void clear();
    /**
     * An accsessor to the card on the top of the pile.
     * @return the card that is on top of the pile.
     */
    const Card& topCard() const;
    /**
     * Gives the number of cards in the pile.
     * @return the number of cards in the pile.
     */
    int size() const;
    /**
     * An accsessor to the card on the bottom of the pile.
     * @return the card that is on the bottom of the pile.
     */
    const Card& bottomCard() const;
    /**
     * Adds a card to the top of the pile.
     * @param card is the card to place on top of the pile.
     */
    void addCard(const Card &card);
    /**
     * Removes a card from the top of the pile.
     * @return the card removed from the pile.
     */
    Card removeCard();
    /**
     * Adds a card to the bottom of the pile.
     * @param card is the card to place at the bottom of the pile.
     */
    void addCardToBottom(const Card &card);
    /**
     * Removes a card from the bottom of the pile.
     * @return the card removed from the pile.
     */
    Card removeCardFromBottom();
    /**
     * Verifies that two card piles are equal.
     * @param rhs is the pile you wish to compare to *this.
     * @note piles are considered to be equal if they have the same cards in the same order.
     * @return true if they are considered equal, false elsewise.
     */
    bool operator==(const CardPile &rhs);
    /**
     * This determines if a pile is empty.
     * @return true if the pile has no cards, false elsewise.
     */
    bool isEmpty() const;
    /**
     * Appends another pile to *this.
     * @param rhs is the pile you wish to place below *this.
     * @return *this.
     */
    CardPile& operator+=(const CardPile &rhs);
    /**
     * Stream insertion operator.
     * @param out is the stream to write the pile of cards into.
     * @note Each card is separated by white space.
     * @param cardPile is the pile to write into the stream.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const CardPile &cardPile);
    /**
     * Stream extraction operator.
     * @param in is the stream to read the pile from.
     * @note Each card is assumed to be separated by white space.
     * @param cardPile is the pile to place the read in cards.
     * @return in.
     */
    friend istream& operator>>(istream &in, CardPile &cardPile);
    /**
     * Stream insertion operator.
     * @param out is the stream to write the pile of cards into.
     * @note Each card is separated by white space.
     * @param pile is the pile to write into the stream.
     * @return out.
     */
    friend QTextStream& operator<<(QTextStream &out, const CardPile &pile);
    /**
     * Stream extraction operator.
     * @param in is the stream to read the pile from.
     * @note Each card is assumed to be separated by white space.
     * @param pile is the pile to place the read in cards.
     * @return in.
     */
    friend QTextStream& operator>>(QTextStream &in, CardPile &pile);

protected:
    /// @param m_pile is a stack of cards.
    CardSequence m_pile;
};
#endif
