/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "card.h"
#include "kardsgterror.h"

#include <qtextstream.h>
#include <qstring.h>
#include <string>
using std::string;

Card::Card():m_rank(RANK_ERR), m_suit(SUIT_ERR)
{}

Card::Card(rank_t rank, suit_t suit):m_rank(rank), m_suit(suit)
{
    if ((rank > RANK_ERR) || (rank < ACE))
        throw KardsGTError(QString("Card"), QString("Card"), QString("The value of rank (%1) is out of scope!").arg(rank));
    if ((suit > SUIT_ERR) || (suit < SPADES))
        throw KardsGTError(QString("Card"), QString("Card"), QString("The value of suit (%1) is out of scope!").arg(suit));
}

Card::suit_t Card::suit() const
{
    return m_suit;
}

Card::rank_t Card::rank() const
{
    return m_rank;
}

void Card::setCard(const QString &card)
{
    // Since we represent all cards as RankSuit they have to be either 2 or 3 in length.
    if ((card.length() != 2) && (card.length() != 3))
    {
        m_rank=Card::RANK_ERR;
        m_suit=Card::SUIT_ERR;
    }
    else
    {
        char rank=card[0], suit=card[1];
        m_rank=generateRank(rank);
        // If the rank was a ten, then the we still have the suit
        if (m_rank == Card::TEN)
            suit=card[2];
        //Determine the suit
        m_suit=generateSuit(suit);
    }
}

void Card::setCard(rank_t rank, suit_t suit)
{
    if ((rank > RANK_ERR) || (rank < ACE))
        throw KardsGTError("Card", "Card", "rank out of scope!");
    if ((suit > SUIT_ERR) || (suit < SPADES))
        throw KardsGTError("Card", "Card", "cardFamily out of scope!");
    m_rank = rank;
    m_suit = suit;
}

Card& Card::operator=(const Card &card)
{
    if (this != &card)
    {
        m_suit=card.suit();
        m_rank=card.rank();
    }
    return *this;
}

int Card::operator+(const Card &card)
{
    return (m_rank + 1) + (card.rank() + 1);
}

int Card::operator-(const Card &card)
{
    return (m_rank + 1) - (card.rank() + 1);
}

bool Card::operator==(const Card &card) const
{
    if ((m_rank == card.rank()) && (m_suit == card.suit()))
        return true;
    else
        return false;
}

bool Card::operator!=(const Card &card) const
{
    if (*this == card)
        return false;
    else
        return true;
}

bool Card::operator<(const Card &card) const
{
    if (m_rank < card.rank())
        return true;
    else
        return false;
}

bool Card::operator>(const Card &card) const
{
    if (m_rank > card.rank())
        return true;
    else
        return false;
}

bool Card::lessThan(const Card &card) const
{
    if (m_suit < card.suit())
        return true;
    else
        return false;
}

ostream& operator<<(ostream &out, const Card &card)
{
    out << card.generateHumanReadable();
    return out;
}

istream& operator>>(istream &in, Card &card)
{
    string cardInput;

    in >> cardInput;
    card.setCard(cardInput);
    return in;
}

QTextStream& operator<<(QTextStream &out, const Card &card)
{
    out << card.generateHumanReadable();
    return out;
}

QTextStream& operator>>(QTextStream &in, Card &card)
{
    QString cardInput;

    in >> cardInput;
    card.setCard(cardInput);
    return in;
}

inline QString Card::generateHumanReadable() const
{
    QString line;

    //Determine the rank
    switch (m_rank)
    {
    case ACE:
        line="A";
        break;
    case TWO:
        line="2";
        break;
    case THREE:
        line="3";
        break;
    case FOUR:
        line="4";
        break;
    case FIVE:
        line="5";
        break;
    case SIX:
        line="6";
        break;
    case SEVEN:
        line="7";
        break;
    case EIGHT:
        line="8";
        break;
    case NINE:
        line="9";
        break;
    case TEN:
        line="10";
        break;
    case JACK:
        line="J";
        break;
    case QUEEN:
        line="Q";
        break;
    case KING:
        line="K";
        break;
    default:
        line="-"; // Means an error in the determination of the rank.
        break;
    }

    //Determine the suit
    switch (m_suit)
    {
    case SPADES:
        line+="S";
        break;
    case HEARTS:
        line+="H";
        break;
    case DIAMONDS:
        line+="D";
        break;
    case CLUBS:
        line+="C";
        break;
    default:
        line+="-"; // Means an error in the determination of the suit.
        break;
    }
    return line;
}

inline Card::rank_t Card::generateRank(char rank) const
{
    switch (toupper(rank))
    {
    case 'A':
        return ACE;
    case '2':
        return TWO;
    case '3':
        return THREE;
    case '4':
        return FOUR;
    case '5':
        return FIVE;
    case '6':
        return SIX;
    case '7':
        return SEVEN;
    case '8':
        return EIGHT;
    case '9':
        return NINE;
    case '1':
        return TEN;
    case 'J':
        return JACK;
    case 'Q':
        return QUEEN;
    case 'K':
        return KING;
    }
    return RANK_ERR;
}

inline Card::suit_t Card::generateSuit(char suit) const
{
    switch (toupper(suit))
    {
    case 'S':
        return SPADES;
    case 'H':
        return HEARTS;
    case 'D':
        return DIAMONDS;
    case 'C':
        return CLUBS;
    }
    return SUIT_ERR;
}

QString Card::toString() const
{
    return generateHumanReadable();
}

bool Card::isEmpty() const
{
    if ((m_rank == RANK_ERR) || (m_suit == SUIT_ERR))
        return true;
    else
        return false;
}
