#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import mediasongwindow as SongWindow

import wx


class SearchWindow(wx.Panel):
    """ Interface to search for a song in the media library. """

    """
     ATTRIBUTES
    """

    """
     EVENT ID
    """
    # Identifier for the area where the user enters search terms.
    ID_SEARCH_ENTRY = wx.NewId()
    # Identifier for the result display.
    ID_SEARCH_RESULTS = wx.NewId() # Display for the results of a search
    # Search for audio media
    __SEARCH_AUDIO = wx.NewId()
    # Search for video media
    __SEARCH_VIDEO = wx.NewId()

    def __init__(self, parent, iconPath, searchAudio=True, searchVideo=True):
        """ Default constructor

         wx.Window parent: is the parent window.
         boolean searchAudio: True if the user wants to search for audio media.
         boolean searchVideo: True if the user wants to search for video media.
        """
        wx.Panel.__init__(self, parent, wx.NewId())
        self.__createControls(searchAudio, searchVideo, iconPath)
        self.__bindEvents()
        self.__doLayout()

    def __createControls(self, searchAudio, searchVideo, iconPath):
        """ Create the controls for the search window.

         boolean searchAudio: True if the user wants to search for audio media.
         boolean searchVideo: True if the user wants to search for video media.
        """
        # Create Search Box
        self.__srchCtrlSearch = wx.SearchCtrl(self, self.ID_SEARCH_ENTRY, \
            style=wx.TE_PROCESS_ENTER)
        self.__srchCtrlSearch.ShowCancelButton(True)
        self.__srchCtrlSearch.ShowSearchButton(True)
        self.__srchCtrlSearch.SetDescriptiveText("Search Media Library")

        # Create Search Box Menu
        self.__mnuSearch = wx.Menu("Search Options")
        self.__mnuSearch.AppendCheckItem(self.__SEARCH_AUDIO, "Audio Media")
        self.__mnuSearch.Check(self.__SEARCH_AUDIO, searchAudio)
        self.__mnuSearch.AppendCheckItem(self.__SEARCH_VIDEO, "Video Media")
        self.__mnuSearch.Check(self.__SEARCH_VIDEO, searchVideo)
        self.__srchCtrlSearch.SetMenu(self.__mnuSearch)

        # Create Search Result Displays
        self.__resultsDisplay = SongWindow.MediaSongWindow(self, iconPath, \
            self.ID_SEARCH_RESULTS)

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        pass

    def __doLayout(self):
        """ Creates a visually pleasing look for the controls.

         All the control objects are placed on the window.
        """

        # Search Entry Layout
        self.__bxSzrSearchEntry = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrSearchEntry.Add(self.__srchCtrlSearch, 1, wx.EXPAND | \
            wx.ALL | wx.FIXED_MINSIZE, 3)

        # Search Result Layout
        self.__bxSzrSearchResults = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrSearchResults.Add(self.__resultsDisplay, 1, \
            wx.EXPAND | wx.ALL, 3)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrInterface.Add(self.__bxSzrSearchEntry, 0, wx.EXPAND | \
            wx.ALL)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__bxSzrSearchResults, 1, wx.EXPAND | \
            wx.ALL)
        self.SetSizer(self.__bxSzrInterface)

    def searchTerms(self):
        """ Accessor to the search terms.

         return string: the search terms entered by the user.
        """
        return self.__srchCtrlSearch.GetValue()

    def searchForAudios(self):
        """ Accessor to the search criteria

         return boolean: True if the user wants to search for audio media.
        """
        return self.__mnuSearch.IsChecked(self.__SEARCH_AUDIO)

    def searchForVideos(self):
        """ Accessor to the search criteria

         return boolean: True if the user wants to search for video media.
        """
        return self.__mnuSearch.IsChecked(self.__SEARCH_VIDEO)

    def clearSearchTerms(self):
        """ Clears the search terms. """
        self.__srchCtrlSearch.SetValue("")

    def addPopUpMenuItem(self, idPopUp, action):
        self.__mnItmPopUp = wx.MenuItem( \
            self.__resultsDisplay.mnDisplayWindowPopup, idPopUp, action)
        self.__resultsDisplay.mnDisplayWindowPopup.AppendItem(self.__mnItmPopUp)

    def selectedSongs(self):
        """ Accessor to the currently selected songs

         return list[MediaSong]: each of the songs selected by the user.
        """
        return self.__resultsDisplay.displayedSongs()

    def clearSearchResults(self):
        """ Clears the display result window """
        self.__resultsDisplay.clearDisplayedSongs()

    def addSearchResult(self, result):
        """ Adds a search result to display for the user.

         MediaSong result: is the song to add.
        """
        self.__resultsDisplay.addDisplayedSong(result)
