#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import wx


class PlayerWindow(wx.Panel):
    """ The interface to control the song playing """

    """
     ATTRIBUTES
    """
    # The lowest possible volume percentage
    __VOLUME_MINIMUM = 0
    # The highest possible volume percentage
    __VOLUME_MAXIMUM = 100
    # The lowest possible position percentage into a song.
    __TRACK_MINIMUM = 0
    # The highest possible position percentage into a song.
    __TRACK_MAXIMUM = 100 # End of song
    # Icon path
    __iconPath = None
    # Volume level prior to toggling mute.
    __preMuteVolume = None
    # Displaying play icon
    __blDisplayingPlay = None

    """
     EVENT ID
    """
    # Used to indicate to play or pause a song.
    ID_PLAY = wx.NewId()
    # Used to indicate to restart the current song.
    ID_RESTART = wx.NewId()
    # Used to indicate to stop a song.
    ID_STOP = wx.NewId()
    # Used to indicate to change the track position.
    ID_TRACK_POSITION = wx.NewId()
    # Used to indicate to change the volume level.
    ID_VOLUME = wx.NewId()
    # Used to Mute/UnMute the volume
    ID_MUTE = wx.NewId()

    def __init__(self, parent, iconPath, id=wx.NewId(), volume=50):
        """ Default constructor

         wx.Window parent: is the parent window.
         wx.WindowID id: is the unique identification number for the window
         int volume: is the percentage volume level to display.
        """
        wx.Panel.__init__(self, parent, id)
        self.__createControls(volume, iconPath)
        self.__bindEvents()
        self.__doLayout()
        self.__iconPath = iconPath
        self.__preMuteVolume = volume
        self.__onChangeVolume(None)

    def __createControls(self, volume, iconPath):
        """ Create the controls for the player window.

         int volume: is the percentage volume level to display.
        """

        # Track Time Information
        self.__sttcTxtTimePosition = wx.StaticText(self, wx.ID_ANY, "Current")
        self.__txtCtrlTimePosition = wx.TextCtrl(self, wx.ID_ANY, "00:00.0", \
            style=wx.TE_READONLY)
        self.__sttcTxtTimeTotal = wx.StaticText(self, wx.ID_ANY, "Total")
        self.__txtCtrlTimeTotal = wx.TextCtrl(self, wx.ID_ANY, "00:00.0", \
            style=wx.TE_READONLY)
        self.__sldrTrackPosition = wx.Slider(self, self.ID_TRACK_POSITION, \
            0, self.__TRACK_MINIMUM, self.__TRACK_MAXIMUM, \
            style=wx.SL_AUTOTICKS)
        self.__sttcTxtSongProgress = wx.StaticText(self, wx.ID_ANY, \
            "Song Progress")

        # Create Player Buttons
        self.__btmpBttnPlay = wx.BitmapButton(self, self.ID_PLAY, \
            wx.Bitmap(iconPath + '/play.png'))
        self.__btmpBttnPlay.SetToolTip(wx.ToolTip( \
            "Play Song From Play-List"))
        self.__blDisplayingPlay = True
        self.__btmpBttnRestart = wx.BitmapButton(self, self.ID_RESTART, \
            wx.Bitmap(iconPath + '/restart.png'))
        self.__btmpBttnRestart.SetToolTip(wx.ToolTip( \
            "Restart Currently Playing Song"))
        self.__btmpBttnRestart.Enable(False)
        self.__btmpBttnStop = wx.BitmapButton(self, self.ID_STOP, \
            wx.Bitmap(iconPath + '/stop.png'))
        self.__btmpBttnStop.SetToolTip(wx.ToolTip( \
            "Stop Currently Playing Song"))
        self.__btmpBttnStop.Enable(False)

        # Volume Control
        self.__sttcTxtSongVolume = wx.StaticText(self, wx.ID_ANY, \
            "Volume")
        self.__sldrVolume = wx.Slider(self, self.ID_VOLUME, volume, \
            self.__VOLUME_MINIMUM, self.__VOLUME_MAXIMUM, \
            style=wx.SL_RIGHT | wx.SL_LABELS)
        self.__btmpBttnMute = wx.BitmapButton(self, self.ID_MUTE, \
            wx.Bitmap(iconPath + '/mute.png'))
        self.__btmpBttnMute.SetToolTip(wx.ToolTip( \
            "Mute Currently Playing Song"))
        self.__btmpBttnMute.Enable(False)

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        wx.EVT_COMMAND_SCROLL(self, self.ID_VOLUME, self.__onChangeVolume)


    def __doLayout(self):
        """ Places the controls in a defined layout. """

        # Player Layout
        self.__bxSzrPlayerControlsRow0 = wx.GridSizer(1, 4)
        self.__bxSzrPlayerControlsRow0.Add(self.__sttcTxtTimePosition, 1, \
            wx.ALL | wx.ALIGN_RIGHT, 1)
        self.__bxSzrPlayerControlsRow0.Add(self.__txtCtrlTimePosition, 1, \
            wx.ALL, 1)
        self.__bxSzrPlayerControlsRow0.Add(self.__btmpBttnPlay, 0, wx.ALL, 1)
        self.__bxSzrPlayerControlsRow0.Add(self.__btmpBttnRestart, 0, \
            wx.ALL, 1)


        self.__bxSzrPlayerControlsRow1 = wx.GridSizer(1, 4)
        self.__bxSzrPlayerControlsRow1.Add(self.__sttcTxtTimeTotal, 1, \
            wx.ALL | wx.ALIGN_RIGHT, 1)
        self.__bxSzrPlayerControlsRow1.Add(self.__txtCtrlTimeTotal, 1, \
            wx.ALL, 1)
        self.__bxSzrPlayerControlsRow1.Add(self.__btmpBttnStop, 0, wx.ALL, 1)

        self.__bxSzrPlayerControls = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrPlayerControls.AddSizer(self.__bxSzrPlayerControlsRow0, \
            0, wx.ALL, 1)
        self.__bxSzrPlayerControls.AddSizer(self.__bxSzrPlayerControlsRow1, \
            0, wx.ALL, 1)
        self.__bxSzrPlayerControls.Add(self.__sttcTxtSongProgress, 0)
        self.__bxSzrPlayerControls.Add(self.__sldrTrackPosition, 1, wx.EXPAND)

        # Volume Layout
        self.__bxSzrVolume = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrVolume.AddStretchSpacer(2)
        self.__bxSzrVolume.Add(self.__sttcTxtSongVolume, 0, wx.ALIGN_CENTER)
        self.__bxSzrVolume.Add(self.__sldrVolume, 0, wx.EXPAND)
        self.__bxSzrVolume.AddStretchSpacer()
        self.__bxSzrVolume.Add(self.__btmpBttnMute, 0, wx.ALL | \
            wx.ALIGN_RIGHT, 1)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrInterface.AddStretchSpacer()
        self.__bxSzrInterface.Add(self.__bxSzrPlayerControls, 0, wx.EXPAND)
        self.__bxSzrInterface.Add(self.__bxSzrVolume, 1, wx.EXPAND | \
            wx.FIXED_MINSIZE)
        self.__bxSzrInterface.AddStretchSpacer()
        self.SetSizer(self.__bxSzrInterface)

    def volumeLevel(self):
        """ Accessor to the displayed volume level """
        return self.__sldrVolume.GetValue()

    def setVolumeLevel(self, volume):
        """ Modifier to the displayed volume level. """
        self.__sldrVolume.SetValue(volume)
        self.__onChangeVolume(None)

    def toggleMute(self):
        """ Toggles the volume between mute and un-mute. """
        if self.volumeLevel() == self.__VOLUME_MINIMUM:
            if self.__preMuteVolume == self.__VOLUME_MINIMUM:
                level = self.__VOLUME_MAXIMUM / 2
            else:
                level = self.__preMuteVolume
        else:
            self.__preMuteVolume = self.volumeLevel()
            level = self.__VOLUME_MINIMUM
        self.setVolumeLevel(level)

    def __onChangeVolume(self, event):
        """ """
        if self.volumeLevel() == self.__VOLUME_MINIMUM:
            self.__btmpBttnMute.SetBitmapLabel(wx.Bitmap(self.__iconPath + \
                "/unmute.png"))
            self.__btmpBttnMute.SetToolTip(wx.ToolTip( \
                "Unmute Currently Playing Song"))
        else:
            self.__btmpBttnMute.SetBitmapLabel(wx.Bitmap(self.__iconPath + \
                "/mute.png"))
            self.__btmpBttnMute.SetToolTip(wx.ToolTip( \
                "Mute Currently Playing Song"))
        if event is not None:
            event.Skip()

    def trackPosition(self):
        """ Accessor to the displayed track position. """
        return self.__sldrTrackPosition.GetValue()

    def setTrackPosition(self, secondsPassed, secondsTotal):
        """ Modifier to the displayed track position.

         float secondsPassed: are the number of seconds into the track.
         flat secondsTotal: are the total number of seconds in the track.
        """
        minutes = secondsPassed / 60
        seconds = float(secondsPassed) % float(60)
        self.__txtCtrlTimePosition.SetValue("%02d:%04.1f" % (minutes, seconds))
        minutes = secondsTotal / 60
        seconds = float(secondsTotal) % float(60)
        self.__txtCtrlTimeTotal.SetValue("%02d:%04.1f" % (minutes, seconds))
        if secondsTotal > 0:
            self.__sldrTrackPosition.SetRange(0, secondsTotal)
            self.__sldrTrackPosition.SetValue(secondsPassed)
        else:
            self.__sldrTrackPosition.SetValue(0)
        if seconds > 0:
            self.__btmpBttnRestart.Enable(True)
            self.__btmpBttnStop.Enable(True)
        else:
            self.__btmpBttnRestart.Enable(False)
            self.__btmpBttnStop.Enable(False)

    def isDisplayingPlay(self):
        """ Determines if the player is displaying "Play".

         return boolean: True if displaying "Play", False else-wise.
        """
        return self.__blDisplayingPlay

    def togglePlay(self):
        """ Toggles between displaying "Play" and "Pause".

         Toggles to "Pause" if currently displaying "Play", and switches to
           "Play" when currently displaying "Pause".
        """
        if self.__blDisplayingPlay:
            self.__btmpBttnPlay.SetBitmapLabel(wx.Bitmap(self.__iconPath + \
                "/pause.png"))
            self.__btmpBttnPlay.SetToolTip(wx.ToolTip( \
                "Pause Currently Playing Song"))
            self.__blDisplayingPlay = False
            self.__btmpBttnMute.Enable(True)
        else:
            self.__btmpBttnPlay.SetBitmapLabel(wx.Bitmap(self.__iconPath + \
                "/play.png"))
            self.__btmpBttnPlay.SetToolTip(wx.ToolTip( \
                "Resume Currently Loaded Song"))
            self.__blDisplayingPlay = True
            self.__btmpBttnMute.Enable(False)
