#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import wx


class MediaLibraryDialogue(wx.Dialog):
    """ A dialogue to configure the media library """

    """
     ATTRIBUTES
    """
    # Indicates if the settings should be saved
    blSaveSettings = False
    # Indicates if the a setting has changed
    __blIsSettingsDirty = False

    """
     Event IDs
    """
    # Used to signal to add a new directory
    __ID_ADD_DIRECTORY = wx.NewId()
    # Used to signal to remove a currently listed directory.
    __ID_REMOVE_DIRECTORY = wx.NewId()

    def __init__(self, parent, directoryPaths=[]):
        """ Dialogue constructor.

         wx.Window parent: is the parent window to the dialogue
         list[string] directoryPaths: the paths to each directory to search \
           for media.
        """
        wx.Dialog.__init__(self, parent, wx.NewId(), "Media Library")
        self.__UpdateControls(directoryPaths)
        self.__bindEvents()
        self.__doLayout()

    def mediaLibraryPaths(self):
        """ Accessor to the paths the user selected

         return list[string]: a list of all the paths the user selected.
        """
        return self.__lstBxDirectories.GetItems()

    def __UpdateControls(self, directoryPaths):
        """ Updates all the user controls for the dialogue.

         list[string] directoryPaths: are the paths to each directory to
           search for media.
        """
        if directoryPaths is None:
            directoryPaths = []

        # Update Directory Listing
        self.__lstBxDirectories = wx.ListBox(self, -1, \
            wx.DefaultPosition, wx.DefaultSize, directoryPaths, wx.LB_SINGLE, \
            wx.DefaultValidator, "Directory List Box")
        self.__bttnAddDirectory = wx.Button(self, \
            self.__ID_ADD_DIRECTORY, "Add Directory")
        self.__bttnRemoveDirectory = wx.Button(self, \
            self.__ID_REMOVE_DIRECTORY, "Remove Directory")

        # Update interface direction buttons
        self.__bttnUpdateLibrary = wx.Button(self, wx.ID_OK, \
            "Update Library")
        self.__bttnClose = wx.Button(self, wx.ID_CANCEL, "Close")

    def __bindEvents(self):
        """ Binds the controls to their corresponding events. """
        wx.EVT_BUTTON(self, self.__ID_ADD_DIRECTORY, \
            self.__onAddDirectory)
        wx.EVT_BUTTON(self, self.__ID_REMOVE_DIRECTORY, \
            self.__onRemoveDirectory)
        wx.EVT_BUTTON(self, wx.ID_OK, self.__onUpdateLibrary)
        wx.EVT_BUTTON(self, wx.ID_CANCEL, self.__onClose)
        wx.EVT_CLOSE(self, self.__onClose)

    def __doLayout(self):
        """ Places the controls in a defined layout. """

        # Directory Listing Layout
        self.__bxSzrDirectories = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrDirectories.Add(self.__lstBxDirectories, 1, \
            wx.EXPAND | wx.ALL, 3)
        self.__bxSzrDirectoryButtons = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrDirectoryButtons.Add(self.__bttnAddDirectory, 0, wx.ALL)
        self.__bxSzrDirectoryButtons.AddSpacer(5)
        self.__bxSzrDirectoryButtons.Add(self.__bttnRemoveDirectory, 0, wx.ALL)
        self.__bxSzrDirectories.Add(self.__bxSzrDirectoryButtons, 0, wx.ALL)

        # Dialogue Buttons
        self.__stdDlgBttnSzrButtons = wx.StdDialogButtonSizer()
        self.__stdDlgBttnSzrButtons.AddButton(self.__bttnUpdateLibrary)
        self.__stdDlgBttnSzrButtons.AddButton(self.__bttnClose)
        self.__stdDlgBttnSzrButtons.Realize()

        # Update Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrInterface.AddSpacer(15)
        self.__bxSzrInterface.Add(self.__bxSzrDirectories, 1, wx.EXPAND | \
            wx.ALL)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__stdDlgBttnSzrButtons, 0, wx.ALL)
        self.__bxSzrInterface.AddSpacer(15)
        self.SetSizer(self.__bxSzrInterface)

    def __onAddDirectory(self, event):
        """ Allows a user to add a directory to the search paths.

         A dialogue for the user to select a directory to add to the media
           search list. If a directory is added, the settings are marked as
           being dirty.
         wx.Event event: is the event object (Not Used)
        """
        drdlgAddDirectory = wx.DirDialog(self, "Choose a directory", "", \
            wx.DD_DIR_MUST_EXIST)
        if (drdlgAddDirectory.ShowModal() == wx.ID_OK):
            path = drdlgAddDirectory.GetPath()
            self.__lstBxDirectories.Append(path)
            self.__blIsSettingsDirty = True

    def __onRemoveDirectory(self, event):
        """ Allows a user to remove a directory from the search paths.

         Removes the selected path from the directory listing. Marks the
           settings as being dirty.
         wx.Event event: is the event object (Not Used)
        """
        index = self.__lstBxDirectories.GetSelection()
        self.__lstBxDirectories.Delete(index)
        self.__blIsSettingsDirty = True

    def __onUpdateLibrary(self, event):
        """ User wants to Update the library.

         Updates a new library database, by marking the settings as dirty.
         wx.Event event: is the event object (Not Used)
        """
        self.blSaveSettings = True
        self.Show(False)
        self.Destroy()

    def __onClose(self, event):
        """ Closes the dialogue.

         Handles the closing of the dialogue. If the settings have been marked
           as being dirty, the user is asked to confirm that they wish to
           discard all of their changes.
         wx.Event event: is the event object (Not Used)
        """
        if self.__blIsSettingsDirty:
            answer = wx.MessageBox("Close and save new paths?", \
                "Media Paths Changed", wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION)
            if answer == wx.YES:
                self.blSaveSettings = True
            elif answer == wx.NO:
                self.blSaveSettings = False
            else:
                return
        self.Show(False)
        self.Destroy()
