#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import os
import fnmatch


class MediaLibraryBase():
    """ Abstract interface for media libraries. """
    # TODO Use a list of the supported media file types, each marked as audio
    #   or video.


    """
     ATTRIBUTES
    """
    # File name to use for all media library databases.
    DATABASE_FILE_NAME = "media_library.database"
    # All the full search paths in the database
    pathSearchList = None
    # The path to the music database
    databasePath = "."

    def searchPaths(self):
        """ Accessor to the paths to search for new media.

         return list[string]: The search paths for new media.
        """
        return self.pathSearchList

    def readSettings(self, configuration):
        """ Reads the media library settings.

         Reads in the database path, and the search paths, with each search
           path separated by a ';'
         SafeConfigParser configuration: holds the media library settings.
        """
        if configuration.has_section('MediaLibrary'):
            if configuration.has_option('MediaLibrary', "databasePath"):
                self.databasePath = configuration.get("MediaLibrary", \
                    "databasePath")
            if configuration.has_option('MediaLibrary', "searchPaths"):
                self.pathSearchList = configuration.get('MediaLibrary', \
                    'searchPaths').split(";")

    def writeSettings(self, configuration):
        """ Writes out the media library settings.

         Sets the database path and the search path with each search path
           separated by a ';'
         SafeConfigParser configuration: is the object to save the settings in.
        """
        if not configuration.has_section('MediaLibrary'):
            configuration.add_section('MediaLibrary')
        configuration.set('MediaLibrary', 'databasePath', self.databasePath)
        # Save the search paths by separating them by ;
        if self.pathSearchList is not None:
            paths = ''
            for path in self.pathSearchList:
                paths += path + ";"
            configuration.set('MediaLibrary', 'searchPaths', paths[:-1])

    def supportedMedia(self):
        """ All the supported media file extensions.

         return list[string]: the list of all media file extensions.
        """
        return ['ogg', 'flac', 'mp3', 'mp2', 'wma', 'wav', 'mov', 'wmv', \
            'avi', 'flv', 'mpeg', 'mpg', 'mpg']

    def locateMediaSongs(self):
        songs = []
        if self.pathSearchList is not None:
            for path in self.pathSearchList:
                print "Checking Path:", path
                uncdPath = path.encode('UTF-8', 'replace')
                for root, dirs, files in os.walk(uncdPath):
                    for name in files:
                        if fnmatch.fnmatch(name.lower(), '*.mp3') or \
                                fnmatch.fnmatch(name.lower(), '*.ogg') or \
                                fnmatch.fnmatch(name.lower(), '*.flac') or \
                                fnmatch.fnmatch(name.lower(), '*.mp2') or \
                                fnmatch.fnmatch(name.lower(), '*.wma') or \
                                fnmatch.fnmatch(name.lower(), '*.wav'):
                            songs.append(("%s/%s" % (root, name), False))
                        elif fnmatch.fnmatch(name.lower(), '*.mov') or \
                                fnmatch.fnmatch(name.lower(), '*.mpg') or \
                                fnmatch.fnmatch(name.lower(), '*.avi') or \
                                fnmatch.fnmatch(name.lower(), '*.flv') or \
                                fnmatch.fnmatch(name.lower(), '*.mpeg') or \
                                fnmatch.fnmatch(name.lower(), '*.asf') or \
                                fnmatch.fnmatch(name.lower(), '*.wmv'):
                            songs.append(("%s/%s" % (root, name), True))
                        else:
                            print "Format not supported: %s/%s" % (root, name)
        return songs

    def buildLibrary(self, paths):
        """ Builds the database of all the audio and video files.

         Creates a new database with the files listed in the search path. If
           a file does not contain an extension of one of the supported media
           formats a message is given to indicate it not being included.
         list[strings] paths: is the paths to the directory to search for all
           audio and video music files.
         return int: the number of songs found
        """
        pass

    def incrementSongPlayed(self, song):
        """ Increments the song counter.

         Increments the song counter by one.
         MediaSong song: is the song that is to be played.
        """
        pass

    def findSongs(self, searchTerms, searchAudio, searchVideo, favourite=None):
        """ Finds songs that meet a given criteria.

         Searches through the database and finds songs that meet all of the
           terms the user entered.
         string searchTerms: is the search items in part or in whole of the
           name of a song either by artist or by title. Each term is separated
           by white space. For inclusion in the results, a song must contain
           all the search terms, case insensitive, and there must be terms to
           compare with.
         boolean searchAudio: True if the song must be an audio media.
         boolean searchVideo: True if the song must be a video media.
         boolean favourite: True if the song must be a favourite, False if it
           must not be a favourite, None if the condition does not matter.
         list[MediaSong]: all the songs found that match the search criteria
           as MediaSong items. If no song is found an empty list is given.
        """
        pass

    def setFavourite(self, song, favourite):
        """ Sets a specific songs favourite status.

         MediaSong song: is the song whose favourite status we are setting.
         boolan favourite: is the favourite status of the song
         boolean: True if the song was found and set, False else-wise.
        """
        pass
