#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
from threading import Lock
from subprocess import Popen, PIPE
import sys


class PlayerController():
    """ Media player controller. """

    """
     ATTRIBUTES
    """
    # Subprocess containing the running player.
    __ppnPlayer = None

    """
     PLAYER THREAD LOCKS
    """
    # Lock for writing to the standard input.
    __lckWriteStandardInput = None
    # Lock for reading from the standard out.
    __lckReadStandardOut = None
    # Lock for reading from the standard error.
    __lckReadStandardError = None


    def __init__(self):
        """ Initialises all locks. """
        print "PlayerController.__init__(self)", self
        self.__lckWriteStandardInput = Lock()
        self.__lckReadStandardOut = Lock()
        self.__lckReadStandardError = Lock()

    def isPlayerReady(self):
        """ Determines if the player is running.

         Indicates if the player is playing a song.
         return bool: true if the player is playing a song, false else-wise.
        """
        if self.__ppnPlayer.poll() is None:
            return False
        else:
            return True

    def startPlayer(self, command):
        """ Starts the player.

         Starts the player by sending a command to the OS command line. This
           will raise an OSError exception if the command fails to execute.
         String command: is the command to issue and start the player.
        """
        self.__ppnPlayer = Popen(command, shell=True, stdin=PIPE, \
            stdout=PIPE, stderr=PIPE)
        if self.__ppnPlayer.poll() is not None:
            raise OSError("Failed to open player controller!")

    def waitForExit(self):
        """ Waits for the open player to close. """
        self.__ppnPlayer.wait()

    def writeStandardInput(self, signal):
        """ Send a signal to the player.

         string signal: is the command to issue to the player on it's
           standard input.
        """
        self.__lckWriteStandardInput.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                self.__ppnPlayer.stdin.write(signal)
            except IOError, message:
                print message
            else:
                print " * PlayerController Sending:%s." % \
                    signal.replace("\n", "")
            finally:
                self.__lckWriteStandardInput.release()
        else:
            self.__lckWriteStandardInput.release()

    def readStandardOut(self):
        """ Read a line from the standard out.

         return string: the first line in standard out.
        """
        line = ""
        self.__lckReadStandardOut.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                line = self.__ppnPlayer.stdout.readline(-1)
            except IOError, message:
                print message
            else:
                print " * PlayerController Standard Out: %s" % \
                    line.replace("\n", "")
            finally:
                self.__lckReadStandardOut.release()
        else:
            self.__lckReadStandardOut.release()
        return line

    def readStandardError(self):
        """ Read a line from the standard error.

         return string: the first line in standard error.
        """
        line = ""
        self.__lckReadStandardError.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                line = self.__ppnPlayer.stderr.readline(-1)
            except IOError, message:
                print message
            else:
                print " * PlayerController Standard Error: %s" % \
                    line.replace("\n", "")
            finally:
                self.__lckReadStandardError.release()
        else:
            self.__lckReadStandardError.release()
        return line
