#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import random

import wx

from mediasong import MediaSong


class PlayListWindow(wx.Panel):
    """ Interface for ordering the songs to be played. """

    """
     ATTRIBUTES
    """
    # A list of MediaSong objects in the order to play each
    __playList = []
    # The width to set the movement buttons at.
    __BUTTON_MOVEMENT_WIDTH = 45
    # The height to set the movement buttons at.
    __BUTTON_MOVEMENT_HEIGHT = 45

    """
     EVENT ID
    """
    # Used to indicate to move the selected song up one position.
    __ID_UP = wx.NewId()
    # Used to indicate to move the selected song down one position.
    __ID_DOWN = wx.NewId()
    # Used to indicate to remove the selected song from the play-list.
    __ID_REMOVE = wx.NewId()

    def __init__(self, parent, id=wx.NewId()):
        """ Default constructor

         wx.Window parent: is the parent window.
         wx.WindowID id: is the unique identification number for the window
        """
        wx.Panel.__init__(self, parent, id)
        random.seed()

        self.__createControls()
        self.__bindEvents()
        self.__doLayout()

    def __createControls(self):
        """ Create the controls for the play-list window. """
        self.__lstBxPlayList = wx.ListBox(self, style=wx.LB_SINGLE)

        # Create Play List Button Movements
        self.__bttnMoveUp = wx.Button(self, self.__ID_UP, "UP", \
            size=wx.Size(self.__BUTTON_MOVEMENT_WIDTH, \
            self.__BUTTON_MOVEMENT_HEIGHT))
        self.__bttnRemove = wx.Button(self, self.__ID_REMOVE, "REM", \
            size=wx.Size(self.__BUTTON_MOVEMENT_WIDTH, \
            self.__BUTTON_MOVEMENT_HEIGHT))
        self.__bttnMoveDown = wx.Button(self, self.__ID_DOWN, "DWN", \
            size=wx.Size(self.__BUTTON_MOVEMENT_WIDTH, \
            self.__BUTTON_MOVEMENT_HEIGHT))

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        wx.EVT_BUTTON(self, self.__ID_UP, self.__onMoveSongUp)
        wx.EVT_BUTTON(self, self.__ID_REMOVE, self.__onRemoveSong)
        wx.EVT_BUTTON(self, self.__ID_DOWN, self.__onMoveSongDown)

    def __doLayout(self):
        """ Creates a visually pleasing look for the controls.

         All the control objects are placed on the window.
        """
        self.__bxSzrPlayList = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrPlayList.Add(self.__lstBxPlayList, 1, wx.EXPAND | \
            wx.ALL, 3)

        # Play List Button Layout
        self.__bxSzrPlayListButtons = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrPlayListButtons.Add(self.__bttnMoveUp, 1, wx.ALL, 3)
        self.__bxSzrPlayListButtons.AddSpacer(1)
        self.__bxSzrPlayListButtons.Add(self.__bttnRemove, 0, wx.ALL, 3)
        self.__bxSzrPlayListButtons.AddSpacer(1)
        self.__bxSzrPlayListButtons.Add(self.__bttnMoveDown, 0, wx.ALL, 3)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrInterface.Add(self.__bxSzrPlayList, 1, wx.EXPAND | wx.ALL)
        self.__bxSzrInterface.Add(self.__bxSzrPlayListButtons, 0, wx.ALL)

        self.SetSizer(self.__bxSzrInterface)

    def addSongsToPlayList(self, songs):
        """ Inserts songs to the end of the play-list.

         list[MediaSong] songs: is the list of songs to add at the end of the
           play-list.
        """
        for song in songs:
            self.__playList.append(song)
            self.__lstBxPlayList.Append("%s - %s" % (song.title, song.artist))

    def currentSelectedSong(self):
        """ Accessor to the currently selected song.

         return MediaSong: the currently selected song
        """
        song = MediaSong()
        index = self.__lstBxPlayList.GetSelection()
        try:
            song = self.__playList[index]
        except IndexError:
            pass
        return song

    def playSelectedSong(self, removeSong=True):
        """ Play the currently selected song.

         Play the currently selected song in the play-list, and then remove it
           from the play-list.
         boolean removeSong: indicates if the selected song should be removed
           after the method returns.
         return MediaSong: the selected song in the play-list.
        """
        selectedSong = MediaSong()
        index = self.__lstBxPlayList.GetSelection()
        try:
            selectedSong = self.__playList[index]
        except IndexError:
            pass
        else:
            if removeSong:
                self.__removeSelectedSong()
        return selectedSong

    def playFirstSong(self, removeSong=True):
        """ Play the first song in the play-list.

         Play the first song in the play-list, and then remove it from the
           play-list.
         boolean removeSong: indicates if the selected song should be removed
           after the method returns.
         return MediaSong: the first song in the play-list.
        """
        self.__lstBxPlayList.SetSelection(0)
        return self.playSelectedSong(removeSong)

    def playRandomSong(self, removeSong=True):
        """ Play a random song from the play-list.

         Play a random song from the play-list, and then remove it from the
           play-list.
         boolean removeSong: indicates if the selected song should be removed
           after the method returns.
         return MediaSong: the randomly selected song from the play-list.
        """
        self.__lstBxPlayList.SetSelection(random.randint(0, \
            len(self.__playList)))
        return self.playSelectedSong(removeSong)

    def __onRemoveSong(self, event):
        """ Removes the selected song from the play-list.

         wx.Event event: is the event object (Not Used)
        """
        self.__removeSelectedSong()

    def __onMoveSongUp(self, event):
        """ Move the selected song up one position in the play-list

         wx.Event event: is the event object (Not Used)
        """
        selectedIndex = self.__lstBxPlayList.GetSelection()

        if selectedIndex > 0:
            self.__swapSongPositions(selectedIndex - 1, selectedIndex, True)

    def __onMoveSongDown(self, event):
        """ Move the selected song down one position in the play-list

         wx.Event event: is the event object (Not Used)
        """
        selectedIndex = self.__lstBxPlayList.GetSelection()

        if selectedIndex < (len(self.__playList) - 1):
            self.__swapSongPositions(selectedIndex, selectedIndex + 1, False)

    def __removeSelectedSong(self):
        """ Removes the selected song from the play-list. """
        if self.__playList:
            selectedIndex = self.__lstBxPlayList.GetSelection()
            self.__lstBxPlayList.Clear()
            tempPlayList = list(self.__playList)
            del self.__playList[:]
            del tempPlayList[selectedIndex]
            self.addSongsToPlayList(tempPlayList)
        if self.__playList:
            self.__lstBxPlayList.SetSelection(selectedIndex)

    def __swapSongPositions(self, frontIndex, backIndex, swapUp):
        """ Switch the position of two songs in the play-list.

         Move the song located at frontIndex to the location of the song at
           backIndex, and the song at backIndex to that of the song at
           frontIndex. Once the swap has been made, set the moved song as
           being selected in the play-list.
         int frontIndex: is the index of the front song to move the position
           of the other song.
         int backIndex: is the index of the song to move to the position of
           the front song.
         boolean swapUp: indicates if we are move the song up or down relative
           to the movement of frontIndex.
        """
        # Set up and clear current play list
        tempPlayList = list(self.__playList)
        self.__lstBxPlayList.Clear()
        del self.__playList[:]

        # Swap the songs
        tempSong = tempPlayList[frontIndex]
        tempPlayList[frontIndex] = tempPlayList[backIndex]
        tempPlayList[backIndex] = tempSong

        # Rebuild the play list
        self.addSongsToPlayList(tempPlayList)
        if swapUp: # frontIndex moved up
            self.__lstBxPlayList.SetSelection(frontIndex)
        else: # frontIndex moved down
            self.__lstBxPlayList.SetSelection(backIndex)
