#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
from threading import Thread, Lock, Condition
from subprocess import Popen, PIPE
import os
import sys
import Queue


class PlayerThread(Thread):
    """ Run a media player in a non-blocking manner. """

    """
     ATTRIBUTES
    """
    # Player command queue
    __commandQueue = Queue.Queue()
    # OS command to start the player.
    __playerCommand = None
    # Player arguments to pass along to the player upon invocation.
    __PlayerCommandArguments = None
    # File path to the currently playing song.
    __songFilePath = None
    # Subprocess containing the running player.
    __ppnPlayer = None
    # Volume level for playing the music
    __volumeLevel = 0
    # Total time in seconds of the currently playing song, in seconds.
    __totalTime = 0.0
    # Current position in the playing song, in seconds.
    __positionTime = 0.0

    """
     PLAYER STATES
    """
    # Indicates if the player is in a starting state.
    __blIsStarting = False
    # Indicates if the player is in a running state.
    __blIsRunning = False
    # Indicates if the player is in a playing state.
    __blIsPlaying = False
    # Indicates if the player is in a paused state.
    __blIsPaused = False
    # Indicates if a song has ended.
    __blIsSongEnded = False

    """
     PLAYER THREAD LOCKS
    """
    # Lock for writing to the standard input.
    __lckWriteStandardInput = None
    # Lock for reading from the standard out.
    __lckReadStandardOut = None
    # Lock for reading from the standard error.
    __lckReadStandardError = None
    # Lock for accessing the command queue.
    __lckCommandQueue = None
    # Lock for running a command from the command queue.
    __lckRunningCommand = None
    # Condition for running a command from the command queue.
    __cndtnRunCommand = None

    """
     PLAYER SIGNALS
    """
    # OS command for playing a song in the player.
    signalPlay = None
    # Player command for pausing a song .
    signalPause = None
    # Player command for un-pausing a song.
    signalUnpause = None
    # Player command for stopping a song.
    signalStop = None
    # Player command for quitting a song.
    signalQuit = None
    # Player command for retrieving the volume level.
    signalVolume = None
    # Player command for retrieving the total time of the song.
    signalTotalTime = None
    # Player command for retrieving the playing position in the song.
    signalPlayerPositionTime = None
    # Player command for setting the playing position in the song.
    signalSetPlayerPositionTime = None

    """
     SIGNAL COMMANDS
    """
    # Indicates a command has not been issued.
    __commandEmpty = hex(0x0)
    # Indicates the player should quit running.
    __commandQuitRunning = hex(0x1)
    # Indicates the player should play the set song.
    __commandPlay = hex(0x2)
    # Indicates the player should pause the playing song.
    __commandPause = hex(0x3)
    # Indicates the player should un-pause the playing song.
    __commandUnpause = hex(0x4)
    # Indicates the player should stop the playing song.
    __commandStop = hex(0x5)
    # Indicates the player should change the volume level.
    __commandVolume = hex(0x6)
    # Indicates the player should set the total time.
    __commandTotalTime = hex(0x7)
    # Indicates the player should set the playing position.
    __commandPositionTime = hex(0x8)
    # Indicates the player should change the playing position.
    __commandSetPositionTime = hex(0x9)
    # Indicates the player has reached the end of a song.
    __commandSongEnd = hex(0xA)

    def __init__(self, playerCommand, arguments, volumeLevel):
        """ Create and start the player thread.

         Creates and starts the player thread. Note the signals to operate
           the player must be set before attempting to use.
         string playerCommand: is the OS command to start the player.
         string arguments: is the command line arguments to pass to the player.
         int volumeLevel: is the percentage volume level for the player.
        """
        Thread.__init__(self, name=playerCommand)
        self.__playerCommand = playerCommand
        self.__PlayerCommandArguments = arguments
        self.__volumeLevel = volumeLevel

        # Create the locks and conditions
        self.__lckWriteStandardInput = Lock()
        self.__lckReadStandardOut = Lock()
        self.__lckReadStandardError = Lock()
        self.__lckRunningCommand = Lock()
        self.__lckCommandQueue = Lock()
        self.__cndtnRunCommand = Condition(self.__lckRunningCommand)
        self.start()

    def run(self):
        """ Process command queue and issue player signals.

         Processes __commandQueue and issue any player signals to the player
           until __commandQuitRunning is placed in the queue and processed.
        """
        self.__blIsRunning = True
        print "Player Thread Started"
        while self.__blIsRunning:
            self.__cndtnRunCommand.acquire()
            command = self.__commandEmpty
            while self.__commandQueue.empty():
                # Determine if the song has ended
                if self.__blIsPlaying and (self.__totalTime > 0.0):
                    if (self.__ppnPlayer.poll() is not None) or \
                            (self.__positionTime == self.__totalTime):
                        self.__pushCommand(self.__commandSongEnd)
                self.__cndtnRunCommand.wait()
            self.__lckCommandQueue.acquire()
            command = self.__commandQueue.get(True)
            self.__lckCommandQueue.release()
            print "\nPlayer Thread Started Processing Command:", command

            # Signal to quit the player
            if command == self.__commandQuitRunning:
                print "Player Thread Running Signal Quit"
                if self.__blIsPlaying or self.__blIsPaused:
                    self.__blIsPlaying = False
                    self.__blIsPaused = False
                    self.__blIsSongEnded = False
                    self.__writeStandardInput('%s\n' % self.signalQuit)
                    self.__ppnPlayer.wait()
                self.__lckCommandQueue.acquire()
                while not self.__commandQueue.empty():
                    self.__commandQueue.get(True)
                self.__lckCommandQueue.release()
                self.__blIsRunning = False
                print "Player Thread End Of Signal Quit"

            # Signal to play the set song
            # TODO use the play signal
            if command == self.__commandPlay:
                self.__blIsStarting = True
                signal = self.__playerCommand + " " + \
                    self.__PlayerCommandArguments + self.__songFilePath
                print "Player Thread Running Signal Play: %s" % signal
                self.__ppnPlayer = Popen(signal, shell=True, \
                    stdin=PIPE, stdout=PIPE, stderr=PIPE)
                if self.__ppnPlayer.poll() is not None:
                    raise OSError("Failed to open media file!")
                self.__blIsPlaying = True
                self.__blIsSongEnded = False
                self.__blIsStarting = False
                print "Player Thread End Of Signal Play"

            # Signal to stop the song
            if command == self.__commandStop:
                print "Player Thread Running Signal Stop: %s" % \
                    self.signalStop
                self.__blIsPlaying = False
                self.__blIsPaused = False
                self.__volumeLevel = 0
                self.__totalTime = 0.0
                self.__positionTime = 0.0
                self.__writeStandardInput('%s\n' % self.signalStop)
                print "Player Thread End Of Signal Stop"

            # Signal end of the song
            if command == self.__commandSongEnd:
                print "Player Thread Running Signal Song End: %s" % \
                    self.signalQuit
                self.__blIsPlaying = False
                self.__blIsPaused = False
                self.__blIsSongEnded = True
                self.__volumeLevel = 0
                self.__totalTime = 0.0
                self.__positionTime = 0.0
                self.__writeStandardInput('%s\n' % self.signalQuit)
                print "Player Thread End Of Signal Song End"

            # Signal to pause the song
            if command == self.__commandPause:
                print "Player Thread Running Signal Pause: %s" % \
                    self.signalPause
                self.__writeStandardInput('%s\n' % self.signalPause)
                print "Player Thread End Of Signal Pause"

            # Signal to unpause the song
            if command == self.__commandUnpause:
                print "Player Thread Running Signal Unpause: %s" % \
                    self.signalUnpause
                self.__writeStandardInput('%s\n' % self.signalUnpause)
                print "Player Thread End Of Signal Unpause"

            # Signal to change the volume of the player
            # TODO make player agnostic using a signal class.
            if command == self.__commandVolume: # set volume
                print "Player Thread Running Signal Volume: %s" % \
                    self.signalVolume
                self.__writeStandardInput('%s %s 1\n' % (self.signalVolume, \
                    self.__volumeLevel))
                print "Player Thread End Of Signal Volume"

            # Signal to get the total time of the song.
            # TODO make player agnostic using a signal class.
            if command == self.__commandTotalTime: # get total time
                print "Player Thread Running Signal TotalTime: %s" % \
                    self.signalTotalTime
                self.__writeStandardInput('%s\n' % self.signalTotalTime)
                line = self.__readStandardOut()
                while line is not "" and not "ANS_LENGTH" in line:
                    line = self.__readStandardOut()
                if line:
                    self.__totalTime = float( \
                        line.split("=")[1].replace("\n", ""))
                print "Player Thread End Of Signal TotalTime"

            # Signal to get the current position of the song.
            # TODO make player agnostic using a signal class.
            if command == self.__commandPositionTime:
                # Check is placed here to ensure we can correctly determine
                #   the end of a song.
                if not self.__blIsPaused:
                    print "Player Thread Running Signal PositionTime: %s" % \
                        self.signalPositionTime
                    self.__writeStandardInput('%s\n' % \
                        self.signalPositionTime)
                    line = self.__readStandardOut()
                    while line is not "" and not \
                            "ANS_TIME_POSITION" in line:
                        line = self.__readStandardOut()
                    if line:
                        self.__positionTime = float( \
                            line.split("=")[1].replace("\n", ""))
                    print "Player Thread End Of Signal PositionTime"

            # Signal to set the current position of the song.
            # TODO make player agnostic using a signal class.
            if command == self.__commandSetPositionTime:
                print "Player Thread Running Signal SetPostionTime %s" % \
                    self.signalSetPositionTime
                self.__writeStandardInput('%s %s 2\n' % \
                    (self.signalSetPositionTime, self.__positionTime))
                print "Player Thread End Of Signal SetPostionTime"
            print "Player Thread Finished Processing Command:", command
            self.__cndtnRunCommand.release()
        print "Player Thread Ended"

    def __pushCommand(self, command):
        """ Adds a signal command to the command queue

         SignalCommand command: is the command to issue as a signal.
        """
        self.__lckCommandQueue.acquire()
        if self.__blIsRunning:
            print "Player Thread Adding Command:", command
            self.__commandQueue.put(command, True)
        self.__lckCommandQueue.release()

    def __writeStandardInput(self, signal):
        """ Send a signal to the player.

         string signal: is the command to issue to the player on it's
           standard input.
        """
        self.__lckWriteStandardInput.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                self.__ppnPlayer.stdin.write(signal)
            except IOError, message:
                print message
            else:
                print "Player Thread Sending Signal:%s." % \
                    signal.replace("\n", "")
            finally:
                self.__lckWriteStandardInput.release()
        else:
            self.__lckWriteStandardInput.release()

    def __readStandardOut(self):
        """ Read a line from the standard out.

         return string: the first line in standard out.
        """
        line = ""
        self.__lckReadStandardOut.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                line = self.__ppnPlayer.stdout.readline(-1)
            except IOError, message:
                print message
            else:
                print "Player Thread Standard Out: %s" % \
                    line.replace("\n", "")
            finally:
                self.__lckReadStandardOut.release()
        else:
            self.__lckReadStandardOut.release()
        return line

    def __readStandardError(self):
        """ Read a line from the standard error.

         return string: the first line in standard error.
        """
        line = ""
        self.__lckReadStandardError.acquire()
        if self.__ppnPlayer.poll() is None:
            try:
                line = self.__ppnPlayer.stderr.readline(-1)
            except IOError, message:
                print message
            else:
                print "Player Thread Standard Error: %s" % \
                    line.replace("\n", "")
            finally:
                self.__lckReadStandardError.release()
        else:
            self.__lckReadStandardError.release()
        return line

    def isRunning(self):
        """ Indicates if the player is in a running state. """
        return self.__blIsRunning

    def isStarting(self):
        """ Indicates if the player is in a starting state. """
        return self.__blIsStarting

    def isSongEnded(self):
        """ Indicates if a song has ended.

         return boolean: True if the player has finished a playing song,
           False else-wise
        """
        return self.__blIsSongEnded

    def isPlaying(self):
        """ Indicates if the player is in a playing state. """
        return self.__blIsPlaying

    def isPaused(self):
        """ Indicates if the player is in a paused state. """
        return self.__blIsPaused

    def quitRunning(self):
        """ Issues the command to quit running. """
        self.__pushCommand(self.__commandQuitRunning)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def play(self, filePath, volume):
        """ Issues the command to play a song.

         string filePath: is the path to the file to play.
         int volume: is the percentage volume level to play the song at.
         return boolean: True if commands are placed in the queue, False
           if a song is already loaded or if the file cannot be found.
        """
        # TODO check to ensure the file exists.
        if not filePath:
            print "PlayerThread.play(self, filePath, volume): Empty filePath"
            return False
        elif not self.__blIsPlaying and not self.__blIsPaused:
            self.__songFilePath = " \"" + filePath + "\""
            self.__pushCommand(self.__commandPlay)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return self.setVolumeLevel(volume, noCheckPlaying=True)
        else:
            print "PlayerThread.play(): Already playing: %s " % \
                self.__songFilePath
            return False

    def stop(self):
        """ Issues the command to stop. """
        if not self.__blIsSongEnded and (self.__blIsPlaying or \
                self.__blIsPaused):
            self.__pushCommand(self.__commandStop)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def pause(self):
        """ Issues the command to pause """
        if self.__blIsPlaying:
            self.__pushCommand(self.__commandPause)
            self.__blIsPlaying = False
            self.__blIsPaused = True
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def unpause(self):
        """ Issues the command to un-pause """
        if self.__blIsPaused:
            self.__pushCommand(self.__commandUnpause)
            self.__blIsPlaying = True
            self.__blIsPaused = False
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def setVolumeLevel(self, volume, noCheckPlaying=False):
        """ Issues the command to set the volume level.

         int volume: is the volume level percentage.
         boolean noCheckPlaying: by passes the requirement to only change the
           volume while the player is playing. Use with caution.
         return boolean: True if the volume command is placed in the queue,
           False if we are not in a playing state.
        """
        if self.__blIsPlaying or noCheckPlaying:
            self.__volumeLevel = volume
            self.__pushCommand(self.__commandVolume)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return True
        else:
            return False

    def volumeLevel(self):
        """ Gives the current volume level percentage. """
        return self.__volumeLevel

    def positionInSong(self):
        """ Give the current position in the playing song.

         Places the command to update the position of the playing song, only
           when in a playing state; condition verified in thread.
         return float: the current seconds into a song.
        """
        self.__pushCommand(self.__commandPositionTime)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()
        return self.__positionTime

    def setPlayPosition(self, seconds):
        """ Issues the command to set the current playing position.

         float seconds: is the number of seconds to play from.
         return boolean: True if we are in a playing state, False else-wise.
        """
        if self.__blIsPlaying:
            self.__positionTime = seconds
            self.__pushCommand(self.__commandSetPositionTime)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return True
        else:
            return False

    def totalSongTime(self):
        """ Give the total time in the playing song.

         Places the command to update the total time of the playing song, only
           when in a playing state.
         return float: the total seconds of the loaded song.
        """
        if self.__blIsPlaying:
            self.__pushCommand(self.__commandTotalTime)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
        return self.__totalTime
